/*********************************************************************/
/*                                                                   */
/*                           SHAvite-3                               */
/*                                                                   */
/* Candidate submission to NIST SHA-3 competition                    */
/*                                                                   */
/* Written by Eli Biham and Orr Dunkelman                            */
/*                                                                   */
/*********************************************************************/

#include <stdio.h>
#include <stdint.h>

#include "shavite3.h"

enum {SUCCESS=0, FAIL=1, BAD_HASHBITLEN=2};

#define __XBIG_ENDIAN	4321
#define __XLITTLE_ENDIAN	1234
#ifdef  WORDS_BIGENDIAN
#define __XBYTE_ORDER	4321
#else
#define __XBYTE_ORDER	1234
#endif

#define	ONE8	0xFFU
#define	ONE16	0xFFFFU

#define	T8(x)	((x) & ONE8)
#define	T16(x)	((x) & ONE16)
#define	T32(x)	((x) & ONE32)

#if __XBYTE_ORDER == __XLITTLE_ENDIAN
#define U8TO32_LITTLE(c)  (((uint32_t*)(c))[0])
#define U16TO8_LITTLE(c, v) ((uint16_t*)(c))[0]=v
#define U32TO8_LITTLE(c, v) ((uint32_t*)(c))[0]=v
#define U64TO8_LITTLE(c, v) ((uint64_t*)(c))[0]=v
#else
#define U8TO32_LITTLE(c)  (((uint32_t)T8(*((uint8_t*)(c)))) | \
			   ((uint32_t)T8(*(((uint8_t*)(c)) + 1)) << 8) | \
			   ((uint32_t)T8(*(((uint8_t*)(c)) + 2)) << 16) | \
			   ((uint32_t)T8(*(((uint8_t*)(c)) + 3)) << 24))
#define U16TO8_LITTLE(c, v)    do { \
		uint16_t tmp_portable_h_x = (v); \
		uint8_t *tmp_portable_h_d = (c); \
		tmp_portable_h_d[0] = T8(tmp_portable_h_x); \
		tmp_portable_h_d[1] = T8(tmp_portable_h_x >> 8); \
	} while (0)
#define U32TO8_LITTLE(c, v)    do { \
		uint32_t tmp_portable_h_x = (v); \
		uint8_t *tmp_portable_h_d = (c); \
		tmp_portable_h_d[0] = T8(tmp_portable_h_x); \
		tmp_portable_h_d[1] = T8(tmp_portable_h_x >> 8); \
		tmp_portable_h_d[2] = T8(tmp_portable_h_x >> 16); \
		tmp_portable_h_d[3] = T8(tmp_portable_h_x >> 24); \
	} while (0)
#define U64TO8_LITTLE(c, v)    do { \
		uint64_t tmp_portable_h_x = (v); \
		uint8_t *tmp_portable_h_d = (c); \
		tmp_portable_h_d[0] = T8(tmp_portable_h_x); \
		tmp_portable_h_d[1] = T8(tmp_portable_h_x >> 8);  \
		tmp_portable_h_d[2] = T8(tmp_portable_h_x >> 16); \
		tmp_portable_h_d[3] = T8(tmp_portable_h_x >> 24); \
		tmp_portable_h_d[4] = T8(tmp_portable_h_x >> 32); \
		tmp_portable_h_d[5] = T8(tmp_portable_h_x >> 40); \
		tmp_portable_h_d[6] = T8(tmp_portable_h_x >> 48); \
		tmp_portable_h_d[7] = T8(tmp_portable_h_x >> 56); \
	} while (0)
#endif

/* ===== "AESround.h" */
/* Tables of the combined SubBytes MixColumns operation */
static const uint32_t Table0[256] = {
0xc66363a5U, 0xf87c7c84U, 0xee777799U, 0xf67b7b8dU, 0xfff2f20dU, 0xd66b6bbdU,
0xde6f6fb1U, 0x91c5c554U, 0x60303050U, 0x02010103U, 0xce6767a9U, 0x562b2b7dU,
0xe7fefe19U, 0xb5d7d762U, 0x4dababe6U, 0xec76769aU, 0x8fcaca45U, 0x1f82829dU,
0x89c9c940U, 0xfa7d7d87U, 0xeffafa15U, 0xb25959ebU, 0x8e4747c9U, 0xfbf0f00bU,
0x41adadecU, 0xb3d4d467U, 0x5fa2a2fdU, 0x45afafeaU, 0x239c9cbfU, 0x53a4a4f7U,
0xe4727296U, 0x9bc0c05bU, 0x75b7b7c2U, 0xe1fdfd1cU, 0x3d9393aeU, 0x4c26266aU,
0x6c36365aU, 0x7e3f3f41U, 0xf5f7f702U, 0x83cccc4fU, 0x6834345cU, 0x51a5a5f4U,
0xd1e5e534U, 0xf9f1f108U, 0xe2717193U, 0xabd8d873U, 0x62313153U, 0x2a15153fU,
0x0804040cU, 0x95c7c752U, 0x46232365U, 0x9dc3c35eU, 0x30181828U, 0x379696a1U,
0x0a05050fU, 0x2f9a9ab5U, 0x0e070709U, 0x24121236U, 0x1b80809bU, 0xdfe2e23dU,
0xcdebeb26U, 0x4e272769U, 0x7fb2b2cdU, 0xea75759fU, 0x1209091bU, 0x1d83839eU,
0x582c2c74U, 0x341a1a2eU, 0x361b1b2dU, 0xdc6e6eb2U, 0xb45a5aeeU, 0x5ba0a0fbU,
0xa45252f6U, 0x763b3b4dU, 0xb7d6d661U, 0x7db3b3ceU, 0x5229297bU, 0xdde3e33eU,
0x5e2f2f71U, 0x13848497U, 0xa65353f5U, 0xb9d1d168U, 0x00000000U, 0xc1eded2cU,
0x40202060U, 0xe3fcfc1fU, 0x79b1b1c8U, 0xb65b5bedU, 0xd46a6abeU, 0x8dcbcb46U,
0x67bebed9U, 0x7239394bU, 0x944a4adeU, 0x984c4cd4U, 0xb05858e8U, 0x85cfcf4aU,
0xbbd0d06bU, 0xc5efef2aU, 0x4faaaae5U, 0xedfbfb16U, 0x864343c5U, 0x9a4d4dd7U,
0x66333355U, 0x11858594U, 0x8a4545cfU, 0xe9f9f910U, 0x04020206U, 0xfe7f7f81U,
0xa05050f0U, 0x783c3c44U, 0x259f9fbaU, 0x4ba8a8e3U, 0xa25151f3U, 0x5da3a3feU,
0x804040c0U, 0x058f8f8aU, 0x3f9292adU, 0x219d9dbcU, 0x70383848U, 0xf1f5f504U,
0x63bcbcdfU, 0x77b6b6c1U, 0xafdada75U, 0x42212163U, 0x20101030U, 0xe5ffff1aU,
0xfdf3f30eU, 0xbfd2d26dU, 0x81cdcd4cU, 0x180c0c14U, 0x26131335U, 0xc3ecec2fU,
0xbe5f5fe1U, 0x359797a2U, 0x884444ccU, 0x2e171739U, 0x93c4c457U, 0x55a7a7f2U,
0xfc7e7e82U, 0x7a3d3d47U, 0xc86464acU, 0xba5d5de7U, 0x3219192bU, 0xe6737395U,
0xc06060a0U, 0x19818198U, 0x9e4f4fd1U, 0xa3dcdc7fU, 0x44222266U, 0x542a2a7eU,
0x3b9090abU, 0x0b888883U, 0x8c4646caU, 0xc7eeee29U, 0x6bb8b8d3U, 0x2814143cU,
0xa7dede79U, 0xbc5e5ee2U, 0x160b0b1dU, 0xaddbdb76U, 0xdbe0e03bU, 0x64323256U,
0x743a3a4eU, 0x140a0a1eU, 0x924949dbU, 0x0c06060aU, 0x4824246cU, 0xb85c5ce4U,
0x9fc2c25dU, 0xbdd3d36eU, 0x43acacefU, 0xc46262a6U, 0x399191a8U, 0x319595a4U,
0xd3e4e437U, 0xf279798bU, 0xd5e7e732U, 0x8bc8c843U, 0x6e373759U, 0xda6d6db7U,
0x018d8d8cU, 0xb1d5d564U, 0x9c4e4ed2U, 0x49a9a9e0U, 0xd86c6cb4U, 0xac5656faU,
0xf3f4f407U, 0xcfeaea25U, 0xca6565afU, 0xf47a7a8eU, 0x47aeaee9U, 0x10080818U,
0x6fbabad5U, 0xf0787888U, 0x4a25256fU, 0x5c2e2e72U, 0x381c1c24U, 0x57a6a6f1U,
0x73b4b4c7U, 0x97c6c651U, 0xcbe8e823U, 0xa1dddd7cU, 0xe874749cU, 0x3e1f1f21U,
0x964b4bddU, 0x61bdbddcU, 0x0d8b8b86U, 0x0f8a8a85U, 0xe0707090U, 0x7c3e3e42U,
0x71b5b5c4U, 0xcc6666aaU, 0x904848d8U, 0x06030305U, 0xf7f6f601U, 0x1c0e0e12U,
0xc26161a3U, 0x6a35355fU, 0xae5757f9U, 0x69b9b9d0U, 0x17868691U, 0x99c1c158U,
0x3a1d1d27U, 0x279e9eb9U, 0xd9e1e138U, 0xebf8f813U, 0x2b9898b3U, 0x22111133U,
0xd26969bbU, 0xa9d9d970U, 0x078e8e89U, 0x339494a7U, 0x2d9b9bb6U, 0x3c1e1e22U,
0x15878792U, 0xc9e9e920U, 0x87cece49U, 0xaa5555ffU, 0x50282878U, 0xa5dfdf7aU,
0x038c8c8fU, 0x59a1a1f8U, 0x09898980U, 0x1a0d0d17U, 0x65bfbfdaU, 0xd7e6e631U,
0x844242c6U, 0xd06868b8U, 0x824141c3U, 0x299999b0U, 0x5a2d2d77U, 0x1e0f0f11U,
0x7bb0b0cbU, 0xa85454fcU, 0x6dbbbbd6U, 0x2c16163aU,
};

static const uint32_t Table1[256] = {
0xa5c66363U, 0x84f87c7cU, 0x99ee7777U, 0x8df67b7bU, 0x0dfff2f2U, 0xbdd66b6bU,
0xb1de6f6fU, 0x5491c5c5U, 0x50603030U, 0x03020101U, 0xa9ce6767U, 0x7d562b2bU,
0x19e7fefeU, 0x62b5d7d7U, 0xe64dababU, 0x9aec7676U, 0x458fcacaU, 0x9d1f8282U,
0x4089c9c9U, 0x87fa7d7dU, 0x15effafaU, 0xebb25959U, 0xc98e4747U, 0x0bfbf0f0U,
0xec41adadU, 0x67b3d4d4U, 0xfd5fa2a2U, 0xea45afafU, 0xbf239c9cU, 0xf753a4a4U,
0x96e47272U, 0x5b9bc0c0U, 0xc275b7b7U, 0x1ce1fdfdU, 0xae3d9393U, 0x6a4c2626U,
0x5a6c3636U, 0x417e3f3fU, 0x02f5f7f7U, 0x4f83ccccU, 0x5c683434U, 0xf451a5a5U,
0x34d1e5e5U, 0x08f9f1f1U, 0x93e27171U, 0x73abd8d8U, 0x53623131U, 0x3f2a1515U,
0x0c080404U, 0x5295c7c7U, 0x65462323U, 0x5e9dc3c3U, 0x28301818U, 0xa1379696U,
0x0f0a0505U, 0xb52f9a9aU, 0x090e0707U, 0x36241212U, 0x9b1b8080U, 0x3ddfe2e2U,
0x26cdebebU, 0x694e2727U, 0xcd7fb2b2U, 0x9fea7575U, 0x1b120909U, 0x9e1d8383U,
0x74582c2cU, 0x2e341a1aU, 0x2d361b1bU, 0xb2dc6e6eU, 0xeeb45a5aU, 0xfb5ba0a0U,
0xf6a45252U, 0x4d763b3bU, 0x61b7d6d6U, 0xce7db3b3U, 0x7b522929U, 0x3edde3e3U,
0x715e2f2fU, 0x97138484U, 0xf5a65353U, 0x68b9d1d1U, 0x00000000U, 0x2cc1ededU,
0x60402020U, 0x1fe3fcfcU, 0xc879b1b1U, 0xedb65b5bU, 0xbed46a6aU, 0x468dcbcbU,
0xd967bebeU, 0x4b723939U, 0xde944a4aU, 0xd4984c4cU, 0xe8b05858U, 0x4a85cfcfU,
0x6bbbd0d0U, 0x2ac5efefU, 0xe54faaaaU, 0x16edfbfbU, 0xc5864343U, 0xd79a4d4dU,
0x55663333U, 0x94118585U, 0xcf8a4545U, 0x10e9f9f9U, 0x06040202U, 0x81fe7f7fU,
0xf0a05050U, 0x44783c3cU, 0xba259f9fU, 0xe34ba8a8U, 0xf3a25151U, 0xfe5da3a3U,
0xc0804040U, 0x8a058f8fU, 0xad3f9292U, 0xbc219d9dU, 0x48703838U, 0x04f1f5f5U,
0xdf63bcbcU, 0xc177b6b6U, 0x75afdadaU, 0x63422121U, 0x30201010U, 0x1ae5ffffU,
0x0efdf3f3U, 0x6dbfd2d2U, 0x4c81cdcdU, 0x14180c0cU, 0x35261313U, 0x2fc3ececU,
0xe1be5f5fU, 0xa2359797U, 0xcc884444U, 0x392e1717U, 0x5793c4c4U, 0xf255a7a7U,
0x82fc7e7eU, 0x477a3d3dU, 0xacc86464U, 0xe7ba5d5dU, 0x2b321919U, 0x95e67373U,
0xa0c06060U, 0x98198181U, 0xd19e4f4fU, 0x7fa3dcdcU, 0x66442222U, 0x7e542a2aU,
0xab3b9090U, 0x830b8888U, 0xca8c4646U, 0x29c7eeeeU, 0xd36bb8b8U, 0x3c281414U,
0x79a7dedeU, 0xe2bc5e5eU, 0x1d160b0bU, 0x76addbdbU, 0x3bdbe0e0U, 0x56643232U,
0x4e743a3aU, 0x1e140a0aU, 0xdb924949U, 0x0a0c0606U, 0x6c482424U, 0xe4b85c5cU,
0x5d9fc2c2U, 0x6ebdd3d3U, 0xef43acacU, 0xa6c46262U, 0xa8399191U, 0xa4319595U,
0x37d3e4e4U, 0x8bf27979U, 0x32d5e7e7U, 0x438bc8c8U, 0x596e3737U, 0xb7da6d6dU,
0x8c018d8dU, 0x64b1d5d5U, 0xd29c4e4eU, 0xe049a9a9U, 0xb4d86c6cU, 0xfaac5656U,
0x07f3f4f4U, 0x25cfeaeaU, 0xafca6565U, 0x8ef47a7aU, 0xe947aeaeU, 0x18100808U,
0xd56fbabaU, 0x88f07878U, 0x6f4a2525U, 0x725c2e2eU, 0x24381c1cU, 0xf157a6a6U,
0xc773b4b4U, 0x5197c6c6U, 0x23cbe8e8U, 0x7ca1ddddU, 0x9ce87474U, 0x213e1f1fU,
0xdd964b4bU, 0xdc61bdbdU, 0x860d8b8bU, 0x850f8a8aU, 0x90e07070U, 0x427c3e3eU,
0xc471b5b5U, 0xaacc6666U, 0xd8904848U, 0x05060303U, 0x01f7f6f6U, 0x121c0e0eU,
0xa3c26161U, 0x5f6a3535U, 0xf9ae5757U, 0xd069b9b9U, 0x91178686U, 0x5899c1c1U,
0x273a1d1dU, 0xb9279e9eU, 0x38d9e1e1U, 0x13ebf8f8U, 0xb32b9898U, 0x33221111U,
0xbbd26969U, 0x70a9d9d9U, 0x89078e8eU, 0xa7339494U, 0xb62d9b9bU, 0x223c1e1eU,
0x92158787U, 0x20c9e9e9U, 0x4987ceceU, 0xffaa5555U, 0x78502828U, 0x7aa5dfdfU,
0x8f038c8cU, 0xf859a1a1U, 0x80098989U, 0x171a0d0dU, 0xda65bfbfU, 0x31d7e6e6U,
0xc6844242U, 0xb8d06868U, 0xc3824141U, 0xb0299999U, 0x775a2d2dU, 0x111e0f0fU,
0xcb7bb0b0U, 0xfca85454U, 0xd66dbbbbU, 0x3a2c1616U,
};

static const uint32_t Table2[256] = {
0x63a5c663U, 0x7c84f87cU, 0x7799ee77U, 0x7b8df67bU, 0xf20dfff2U, 0x6bbdd66bU,
0x6fb1de6fU, 0xc55491c5U, 0x30506030U, 0x01030201U, 0x67a9ce67U, 0x2b7d562bU,
0xfe19e7feU, 0xd762b5d7U, 0xabe64dabU, 0x769aec76U, 0xca458fcaU, 0x829d1f82U,
0xc94089c9U, 0x7d87fa7dU, 0xfa15effaU, 0x59ebb259U, 0x47c98e47U, 0xf00bfbf0U,
0xadec41adU, 0xd467b3d4U, 0xa2fd5fa2U, 0xafea45afU, 0x9cbf239cU, 0xa4f753a4U,
0x7296e472U, 0xc05b9bc0U, 0xb7c275b7U, 0xfd1ce1fdU, 0x93ae3d93U, 0x266a4c26U,
0x365a6c36U, 0x3f417e3fU, 0xf702f5f7U, 0xcc4f83ccU, 0x345c6834U, 0xa5f451a5U,
0xe534d1e5U, 0xf108f9f1U, 0x7193e271U, 0xd873abd8U, 0x31536231U, 0x153f2a15U,
0x040c0804U, 0xc75295c7U, 0x23654623U, 0xc35e9dc3U, 0x18283018U, 0x96a13796U,
0x050f0a05U, 0x9ab52f9aU, 0x07090e07U, 0x12362412U, 0x809b1b80U, 0xe23ddfe2U,
0xeb26cdebU, 0x27694e27U, 0xb2cd7fb2U, 0x759fea75U, 0x091b1209U, 0x839e1d83U,
0x2c74582cU, 0x1a2e341aU, 0x1b2d361bU, 0x6eb2dc6eU, 0x5aeeb45aU, 0xa0fb5ba0U,
0x52f6a452U, 0x3b4d763bU, 0xd661b7d6U, 0xb3ce7db3U, 0x297b5229U, 0xe33edde3U,
0x2f715e2fU, 0x84971384U, 0x53f5a653U, 0xd168b9d1U, 0x00000000U, 0xed2cc1edU,
0x20604020U, 0xfc1fe3fcU, 0xb1c879b1U, 0x5bedb65bU, 0x6abed46aU, 0xcb468dcbU,
0xbed967beU, 0x394b7239U, 0x4ade944aU, 0x4cd4984cU, 0x58e8b058U, 0xcf4a85cfU,
0xd06bbbd0U, 0xef2ac5efU, 0xaae54faaU, 0xfb16edfbU, 0x43c58643U, 0x4dd79a4dU,
0x33556633U, 0x85941185U, 0x45cf8a45U, 0xf910e9f9U, 0x02060402U, 0x7f81fe7fU,
0x50f0a050U, 0x3c44783cU, 0x9fba259fU, 0xa8e34ba8U, 0x51f3a251U, 0xa3fe5da3U,
0x40c08040U, 0x8f8a058fU, 0x92ad3f92U, 0x9dbc219dU, 0x38487038U, 0xf504f1f5U,
0xbcdf63bcU, 0xb6c177b6U, 0xda75afdaU, 0x21634221U, 0x10302010U, 0xff1ae5ffU,
0xf30efdf3U, 0xd26dbfd2U, 0xcd4c81cdU, 0x0c14180cU, 0x13352613U, 0xec2fc3ecU,
0x5fe1be5fU, 0x97a23597U, 0x44cc8844U, 0x17392e17U, 0xc45793c4U, 0xa7f255a7U,
0x7e82fc7eU, 0x3d477a3dU, 0x64acc864U, 0x5de7ba5dU, 0x192b3219U, 0x7395e673U,
0x60a0c060U, 0x81981981U, 0x4fd19e4fU, 0xdc7fa3dcU, 0x22664422U, 0x2a7e542aU,
0x90ab3b90U, 0x88830b88U, 0x46ca8c46U, 0xee29c7eeU, 0xb8d36bb8U, 0x143c2814U,
0xde79a7deU, 0x5ee2bc5eU, 0x0b1d160bU, 0xdb76addbU, 0xe03bdbe0U, 0x32566432U,
0x3a4e743aU, 0x0a1e140aU, 0x49db9249U, 0x060a0c06U, 0x246c4824U, 0x5ce4b85cU,
0xc25d9fc2U, 0xd36ebdd3U, 0xacef43acU, 0x62a6c462U, 0x91a83991U, 0x95a43195U,
0xe437d3e4U, 0x798bf279U, 0xe732d5e7U, 0xc8438bc8U, 0x37596e37U, 0x6db7da6dU,
0x8d8c018dU, 0xd564b1d5U, 0x4ed29c4eU, 0xa9e049a9U, 0x6cb4d86cU, 0x56faac56U,
0xf407f3f4U, 0xea25cfeaU, 0x65afca65U, 0x7a8ef47aU, 0xaee947aeU, 0x08181008U,
0xbad56fbaU, 0x7888f078U, 0x256f4a25U, 0x2e725c2eU, 0x1c24381cU, 0xa6f157a6U,
0xb4c773b4U, 0xc65197c6U, 0xe823cbe8U, 0xdd7ca1ddU, 0x749ce874U, 0x1f213e1fU,
0x4bdd964bU, 0xbddc61bdU, 0x8b860d8bU, 0x8a850f8aU, 0x7090e070U, 0x3e427c3eU,
0xb5c471b5U, 0x66aacc66U, 0x48d89048U, 0x03050603U, 0xf601f7f6U, 0x0e121c0eU,
0x61a3c261U, 0x355f6a35U, 0x57f9ae57U, 0xb9d069b9U, 0x86911786U, 0xc15899c1U,
0x1d273a1dU, 0x9eb9279eU, 0xe138d9e1U, 0xf813ebf8U, 0x98b32b98U, 0x11332211U,
0x69bbd269U, 0xd970a9d9U, 0x8e89078eU, 0x94a73394U, 0x9bb62d9bU, 0x1e223c1eU,
0x87921587U, 0xe920c9e9U, 0xce4987ceU, 0x55ffaa55U, 0x28785028U, 0xdf7aa5dfU,
0x8c8f038cU, 0xa1f859a1U, 0x89800989U, 0x0d171a0dU, 0xbfda65bfU, 0xe631d7e6U,
0x42c68442U, 0x68b8d068U, 0x41c38241U, 0x99b02999U, 0x2d775a2dU, 0x0f111e0fU,
0xb0cb7bb0U, 0x54fca854U, 0xbbd66dbbU, 0x163a2c16U,
};

static const uint32_t Table3[256] = {
0x6363a5c6U, 0x7c7c84f8U, 0x777799eeU, 0x7b7b8df6U, 0xf2f20dffU, 0x6b6bbdd6U,
0x6f6fb1deU, 0xc5c55491U, 0x30305060U, 0x01010302U, 0x6767a9ceU, 0x2b2b7d56U,
0xfefe19e7U, 0xd7d762b5U, 0xababe64dU, 0x76769aecU, 0xcaca458fU, 0x82829d1fU,
0xc9c94089U, 0x7d7d87faU, 0xfafa15efU, 0x5959ebb2U, 0x4747c98eU, 0xf0f00bfbU,
0xadadec41U, 0xd4d467b3U, 0xa2a2fd5fU, 0xafafea45U, 0x9c9cbf23U, 0xa4a4f753U,
0x727296e4U, 0xc0c05b9bU, 0xb7b7c275U, 0xfdfd1ce1U, 0x9393ae3dU, 0x26266a4cU,
0x36365a6cU, 0x3f3f417eU, 0xf7f702f5U, 0xcccc4f83U, 0x34345c68U, 0xa5a5f451U,
0xe5e534d1U, 0xf1f108f9U, 0x717193e2U, 0xd8d873abU, 0x31315362U, 0x15153f2aU,
0x04040c08U, 0xc7c75295U, 0x23236546U, 0xc3c35e9dU, 0x18182830U, 0x9696a137U,
0x05050f0aU, 0x9a9ab52fU, 0x0707090eU, 0x12123624U, 0x80809b1bU, 0xe2e23ddfU,
0xebeb26cdU, 0x2727694eU, 0xb2b2cd7fU, 0x75759feaU, 0x09091b12U, 0x83839e1dU,
0x2c2c7458U, 0x1a1a2e34U, 0x1b1b2d36U, 0x6e6eb2dcU, 0x5a5aeeb4U, 0xa0a0fb5bU,
0x5252f6a4U, 0x3b3b4d76U, 0xd6d661b7U, 0xb3b3ce7dU, 0x29297b52U, 0xe3e33eddU,
0x2f2f715eU, 0x84849713U, 0x5353f5a6U, 0xd1d168b9U, 0x00000000U, 0xeded2cc1U,
0x20206040U, 0xfcfc1fe3U, 0xb1b1c879U, 0x5b5bedb6U, 0x6a6abed4U, 0xcbcb468dU,
0xbebed967U, 0x39394b72U, 0x4a4ade94U, 0x4c4cd498U, 0x5858e8b0U, 0xcfcf4a85U,
0xd0d06bbbU, 0xefef2ac5U, 0xaaaae54fU, 0xfbfb16edU, 0x4343c586U, 0x4d4dd79aU,
0x33335566U, 0x85859411U, 0x4545cf8aU, 0xf9f910e9U, 0x02020604U, 0x7f7f81feU,
0x5050f0a0U, 0x3c3c4478U, 0x9f9fba25U, 0xa8a8e34bU, 0x5151f3a2U, 0xa3a3fe5dU,
0x4040c080U, 0x8f8f8a05U, 0x9292ad3fU, 0x9d9dbc21U, 0x38384870U, 0xf5f504f1U,
0xbcbcdf63U, 0xb6b6c177U, 0xdada75afU, 0x21216342U, 0x10103020U, 0xffff1ae5U,
0xf3f30efdU, 0xd2d26dbfU, 0xcdcd4c81U, 0x0c0c1418U, 0x13133526U, 0xecec2fc3U,
0x5f5fe1beU, 0x9797a235U, 0x4444cc88U, 0x1717392eU, 0xc4c45793U, 0xa7a7f255U,
0x7e7e82fcU, 0x3d3d477aU, 0x6464acc8U, 0x5d5de7baU, 0x19192b32U, 0x737395e6U,
0x6060a0c0U, 0x81819819U, 0x4f4fd19eU, 0xdcdc7fa3U, 0x22226644U, 0x2a2a7e54U,
0x9090ab3bU, 0x8888830bU, 0x4646ca8cU, 0xeeee29c7U, 0xb8b8d36bU, 0x14143c28U,
0xdede79a7U, 0x5e5ee2bcU, 0x0b0b1d16U, 0xdbdb76adU, 0xe0e03bdbU, 0x32325664U,
0x3a3a4e74U, 0x0a0a1e14U, 0x4949db92U, 0x06060a0cU, 0x24246c48U, 0x5c5ce4b8U,
0xc2c25d9fU, 0xd3d36ebdU, 0xacacef43U, 0x6262a6c4U, 0x9191a839U, 0x9595a431U,
0xe4e437d3U, 0x79798bf2U, 0xe7e732d5U, 0xc8c8438bU, 0x3737596eU, 0x6d6db7daU,
0x8d8d8c01U, 0xd5d564b1U, 0x4e4ed29cU, 0xa9a9e049U, 0x6c6cb4d8U, 0x5656faacU,
0xf4f407f3U, 0xeaea25cfU, 0x6565afcaU, 0x7a7a8ef4U, 0xaeaee947U, 0x08081810U,
0xbabad56fU, 0x787888f0U, 0x25256f4aU, 0x2e2e725cU, 0x1c1c2438U, 0xa6a6f157U,
0xb4b4c773U, 0xc6c65197U, 0xe8e823cbU, 0xdddd7ca1U, 0x74749ce8U, 0x1f1f213eU,
0x4b4bdd96U, 0xbdbddc61U, 0x8b8b860dU, 0x8a8a850fU, 0x707090e0U, 0x3e3e427cU,
0xb5b5c471U, 0x6666aaccU, 0x4848d890U, 0x03030506U, 0xf6f601f7U, 0x0e0e121cU,
0x6161a3c2U, 0x35355f6aU, 0x5757f9aeU, 0xb9b9d069U, 0x86869117U, 0xc1c15899U,
0x1d1d273aU, 0x9e9eb927U, 0xe1e138d9U, 0xf8f813ebU, 0x9898b32bU, 0x11113322U,
0x6969bbd2U, 0xd9d970a9U, 0x8e8e8907U, 0x9494a733U, 0x9b9bb62dU, 0x1e1e223cU,
0x87879215U, 0xe9e920c9U, 0xcece4987U, 0x5555ffaaU, 0x28287850U, 0xdfdf7aa5U,
0x8c8c8f03U, 0xa1a1f859U, 0x89898009U, 0x0d0d171aU, 0xbfbfda65U, 0xe6e631d7U,
0x4242c684U, 0x6868b8d0U, 0x4141c382U, 0x9999b029U, 0x2d2d775aU, 0x0f0f111eU,
0xb0b0cb7bU, 0x5454fca8U, 0xbbbbd66dU, 0x16163a2cU,
};

#ifdef	UNUSED
/* One round of AES                                         */
/* Input - 4 input words  			            */
/* The input passes through ShiftRows, SubBytes, MixColumns */
/* Then XORed with the round subkey to produce output	    */

static
void roundAES(const uint32_t input[4], uint32_t output[4], const uint32_t key [4])
{
   output[0] = Table0[input[0] >> 24] ^ Table1[(input[1] >> 16) & 0xff] ^ \
	       Table2[(input[2] >> 8) & 0xff] ^ Table3[input[3] & 0xff]^key[0];
   output[1] = Table0[input[1] >> 24] ^ Table1[(input[2] >> 16) & 0xff] ^ \
	       Table2[(input[3] >> 8) & 0xff] ^ Table3[input[0] & 0xff]^key[1];
   output[2] = Table0[input[2] >> 24] ^ Table1[(input[3] >> 16) & 0xff] ^ \
	       Table2[(input[0] >> 8) & 0xff] ^ Table3[input[1] & 0xff]^key[2];
   output[3] = Table0[input[3] >> 24] ^ Table1[(input[0] >> 16) & 0xff] ^ \
	       Table2[(input[1] >> 8) & 0xff] ^ Table3[input[2] & 0xff]^key[3];
}
#endif	/* UNUSED */


/* One round of AES                                         */
/* Input - 4 input words  			            */
/* First the input is XORed with the key, and only then     */
/* passes through ShiftRows, SubBytes, MixColumns 	    */

static
void roundAESkeyfirst(uint32_t input[4], uint32_t output[4], const uint32_t key [4])
{
   int i;

/* XORing the input with the key */
   for (i=0;i<4;i++) 
      input[i]^=key[i];

   output[0] = Table0[input[0] >> 24] ^ Table1[(input[1] >> 16) & 0xff] ^ \
	       Table2[(input[2] >> 8) & 0xff] ^ Table3[input[3] & 0xff]; 
   output[1] = Table0[input[1] >> 24] ^ Table1[(input[2] >> 16) & 0xff] ^ \
	       Table2[(input[3] >> 8) & 0xff] ^ Table3[input[0] & 0xff];
   output[2] = Table0[input[2] >> 24] ^ Table1[(input[3] >> 16) & 0xff] ^ \
	       Table2[(input[0] >> 8) & 0xff] ^ Table3[input[1] & 0xff];
   output[3] = Table0[input[3] >> 24] ^ Table1[(input[0] >> 16) & 0xff] ^ \
	       Table2[(input[1] >> 8) & 0xff] ^ Table3[input[2] & 0xff];
}
/* ===== */

/* ===== "SHAvite3-256.h" */
#define InternalRounds256 3
#define ExternalRounds256 12
#define ExpandedMessageSize256 (ExternalRounds256*4*(InternalRounds256))

/* The message expansion takes a 16 words (stored in rk[0,.,15]) and */
/* the salt/counter values to produce the full expanded message      */

static
void MessageExpansion256 (uint32_t rk[ExpandedMessageSize256], uint32_t salt[8],
			 uint32_t counter[2])
{
   int i,j,k;
   uint32_t input[4],output[4];

/* There are four iterations of the message expansion steps          */
   for (k=0;k<4;k++) 
   {

/* The nonlinear expansion step  				     */

      for (i=16+32*k;i<32+32*k;)
      {
         input[0]=rk[i-15];
         input[1]=rk[i-14];
         input[2]=rk[i-13];
         input[3]=rk[i-16];

/* Round of AES using salt[0..3] as the masking subkey               */
         roundAESkeyfirst(input,output,salt);

         for (j=0;j<4;j++)
            rk[i+j] = output[j] ^ rk[i+j-4];

/* Mixing the counter 						     */

	 if (i==16) 
         {
	    rk[16] ^= counter[0];
	    rk[17] ^= counter[1];
         }
	 if (i==84) 
	 {
	    rk[86] ^= counter[1];
	    rk[87] ^= counter[0];
         }

         i += 4;

         input[0]=rk[i-15];
         input[1]=rk[i-14];
         input[2]=rk[i-13];
         input[3]=rk[i-16];

/* Round of AES using salt[4..7] as the masking subkey               */
         roundAESkeyfirst(input,output,salt+4);

         for (j=0;j<4;j++)
            rk[i+j] = output[j] ^ rk[i+j-4];

/* Mixing the counter 						     */
	 if (i==56) 
	 {
	    rk[57] ^= counter[1];
	    rk[58] ^= counter[0];
         }
	 if (i==124) 
	 {
	    rk[124] ^= counter[0];
	    rk[127] ^= counter[1];
         }
         i += 4;
      }

/* The linear expansion step      				     */

      for (i=32+32*k;i<48+32*k;i++)
         rk[i] = rk[i-16]^rk[i-3];
   }

   return;
}

/* Encrypts the plaintext pt[] using the key message[], salt[],      */
/* and counter[], to produce the ciphertext ct[]                     */

static
void E256(uint32_t pt[8], uint32_t ct[8], uint32_t message[16], uint32_t salt[8], uint32_t counter[2])
{
   uint32_t state[8],i,j;
   uint32_t input[4],output[4],temp;
   uint32_t rk[ExpandedMessageSize256];

/* Setting the internal state 					     */

   for (i=0;i<8;i++)
      state[i]=pt[i]; 

  
/* Initializing the rk[] array with the message words		     */

   for (i=0;i<16;i++) 
      rk[i]=message[i];

   
/* Expanding the message 					     */

   MessageExpansion256(rk,salt,counter);
   
    
/* Repeat the round function 12 times 				     */

   for (i=0;i<12;i++)
   {

     
/* setting the round function input                                  */

      for (j=0;j<4;j++) 
         input[j]=state[4+j];


/* F_3 performs 3 AES rounds 					     */
      roundAESkeyfirst(input,output,rk+12*i);
      for (j=0;j<4;j++)
	 input[j]=output[j];
      roundAESkeyfirst(input,output,rk+12*i+4);
      for (j=0;j<4;j++)
	 input[j]=output[j];
      roundAESkeyfirst(input,output,rk+12*i+8);


/* XORing the output of the F3 to the left half                      */

      for (j=0;j<4;j++)
	 state[j]^=output[j];


/* Swapping the two halves 					     */

      for (j=0;j<4;j++) {
	 temp = state[j];
	 state[j] = state[j+4];
	 state[j+4] = temp;
      }
   }


/* Copying the ciphertext to the output				     */

   for (i=0;i<8;i++) 
      ct[i]=state[i]; 
 
 
   return;
}

/* The actual compression function C_{256}                           */

static
void Compress256(const uint8_t *message_block, uint8_t *chaining_value, uint64_t counter,
	      const uint8_t salt[32])
{    
   uint32_t pt[8],ct[8];
   uint32_t msg_u32[16];
   uint32_t salt_u32[8];
   uint32_t cnt[2];
   int i;

/* Translating all the inputs to 32-bit words			     */

   for (i=0;i<8;i++)
      pt[i]=U8TO32_LITTLE(chaining_value+4*i);

   for (i=0;i<16;i++)
      msg_u32[i]=U8TO32_LITTLE(message_block+4*i);

   for (i=0;i<8;i++)
      salt_u32[i]=U8TO32_LITTLE(salt+4*i);
   
   cnt[1]=(uint32_t)(counter>>32);
   cnt[0]=(uint32_t)(counter & 0xFFFFFFFFULL);


/* Computing the encryption function				     */

   E256(pt, ct, msg_u32, salt_u32, cnt);


/* Davies-Meyer transformation 					     */
   
   for (i=0; i<8; i++)
       pt[i]^=ct[i];


/* Translating the output to 8-bit words			     */

   for (i=0; i<8; i++)
       U32TO8_LITTLE(chaining_value+i*4, pt[i]);

   return;
}
/* ===== */

/* ===== "SHAvite3-512.h" */
#define InternalRounds512 4
#define ExternalRounds512 14
#define ExpandedMessageSize512 (ExternalRounds512*4*(InternalRounds512)*3)

/* The message expansion takes a 16 words (stored in rk[0,.,15]) and */
/* the salt/counter values to produce the full expanded message      */

static
void MessageExpansion512 (uint32_t rk[ExpandedMessageSize512], uint32_t salt[16],
			 uint32_t counter[4])
{
   int i,j,k;
   uint32_t input[4],output[4];

/* There are six full iterations of the message expansion steps      */

   for (k=0;k<7;k++) 
      {

/* The nonlinear expansion step                                      */

      for (i=32+64*k;i<64+64*k;)
      {
         input[0]=rk[i-31];
         input[1]=rk[i-30];
         input[2]=rk[i-29];
         input[3]=rk[i-32];

/* Round of AES using salt[0..3] as the masking subkey               */
         roundAESkeyfirst(input,output,salt);

         for (j=0;j<4;j++)
            rk[i+j] = output[j] ^ rk[i+j-4];

/* Mixing the counter                                                */
	 if (i==32) 
	 {
	    rk[32] ^= counter[0];
	    rk[33] ^= counter[1];
	    rk[34] ^= counter[2];
	    rk[35] ^= counter[3];
         }

         i += 4;

         input[0]=rk[i-31];
         input[1]=rk[i-30];
         input[2]=rk[i-29];
         input[3]=rk[i-32];

/* Round of AES using salt[4..7] as the masking subkey               */
         roundAESkeyfirst(input,output,salt+4);

         for (j=0;j<4;j++)
            rk[i+j] = output[j] ^ rk[i+j-4];

/* Mixing the counter                                                */
	 if (i==164) 
	 {
	    rk[164] ^= counter[3];
	    rk[165] ^= counter[2];
	    rk[166] ^= counter[1];
	    rk[167] ^= counter[0];
         }
         i += 4;

         input[0]=rk[i-31];
         input[1]=rk[i-30];
         input[2]=rk[i-29];
         input[3]=rk[i-32];

/* Round of AES using salt[8..11] as the masking subkey              */
         roundAESkeyfirst(input,output,salt+8);

         for (j=0;j<4;j++)
            rk[i+j] = output[j] ^ rk[i+j-4];

/* Mixing the counter                                                */

	 if (i==440) 
	 {
	    rk[440] ^= counter[1];
	    rk[441] ^= counter[0];
	    rk[442] ^= counter[3];
	    rk[443] ^= counter[2];
         }

         i += 4;

         input[0]=rk[i-31];
         input[1]=rk[i-30];
         input[2]=rk[i-29];
         input[3]=rk[i-32];

/* Round of AES using salt[12..15] as the masking subkey             */
         roundAESkeyfirst(input,output,salt+12);

         for (j=0;j<4;j++)
            rk[i+j] = output[j] ^ rk[i+j-4];

/* Mixing the counter                                                */
	 if (i==316) 
	 {
	    rk[316] ^= counter[2];
	    rk[317] ^= counter[3];
	    rk[318] ^= counter[0];
	    rk[319] ^= counter[1];
         }
         i += 4;
      }

/* The linear expansion step is done only six times                  */

      if (k!=6)
         for (i=64+64*k;i<96+64*k;i++)
            rk[i] = rk[i-32]^rk[i-7];
   }

   return;
}

/* Encrypts the plaintext pt[] using the key message[], salt[],      */
/* and counter[], to produce the ciphertext ct[]                     */

static
void E512(uint32_t pt[16], uint32_t ct[16], uint32_t message[32], uint32_t salt[16], uint32_t counter[4])
{
   uint32_t state[16],i,j;
   uint32_t input[4],output[4],temp;
   uint32_t rk[ExpandedMessageSize512];

/* Setting the internal state                                        */

   for (i=0;i<16;i++) 
      state[i]=pt[i]; 

  
/* Initializing the rk[] array with the message words                */

   for (i=0;i<32;i++) 
      rk[i]=message[i];


/* Expanding the message                                             */

   MessageExpansion512(rk,salt,counter);


/* Repeat the round function 12 times                                */

   for (i=0;i<14;i++)
   {

/* setting the round function input                                  */

      for (j=0;j<4;j++) 
          input[j]=state[4+j];


/* F_4 performs 4 AES rounds                                         */

      roundAESkeyfirst(input,output,rk+32*i);
      for (j=0;j<4;j++)
	 input[j]=output[j];
      roundAESkeyfirst(input,output,rk+32*i+4);
      for (j=0;j<4;j++)
	 input[j]=output[j];
      roundAESkeyfirst(input,output,rk+32*i+8);
      for (j=0;j<4;j++)
	 input[j]=output[j];
      roundAESkeyfirst(input,output,rk+32*i+12);


/* XORing the output of the F4 to the next 128-bit word 	     */

      for (j=0;j<4;j++)
	 state[j]^=output[j];


/* Apply F_4 again (to the third 128-bit word)                       */

      for(j=0;j<4;j++) 
         input[j]=state[12+j];


/* F_4 performs 4 AES rounds                                         */

      roundAESkeyfirst(input,output,rk+32*i+16);
      for(j=0;j<4;j++)
	 input[j]=output[j];
      roundAESkeyfirst(input,output,rk+32*i+20);
      for(j=0;j<4;j++)
	 input[j]=output[j];
      roundAESkeyfirst(input,output,rk+32*i+24);
      for(j=0;j<4;j++)
	 input[j]=output[j];
      roundAESkeyfirst(input,output,rk+32*i+28);


/* XORing the output of the F4 to the next 128-bit word 	     */

      for (j=0;j<4;j++)
	 state[8+j]^=output[j];


/* Rotating the four words 					     */

      for (j=0;j<4;j++) 
      {
	 temp = state[j];
	 state[j] = state[j+4];
	 state[j+4] = state[j+8];
	 state[j+8] = state[j+12];
	 state[j+12] = temp;
      }
   }


/* Copying the ciphertext to the output                              */

   for (i=0;i<16;i++) 
      ct[i]=state[i]; 

   return;
}

/* The actual compression function C_{512}                           */

static
void Compress512(const uint8_t *message_block, uint8_t *chaining_value, uint64_t counter,
	      const uint8_t salt[64])
{    
   uint32_t pt[16],ct[16];
   uint32_t msg_u32[32];
   uint32_t salt_u32[16];
   uint32_t cnt[4];
   int i;

/* Translating all the inputs to 32-bit words                        */

   for (i=0;i<16;i++)
      pt[i]=U8TO32_LITTLE(chaining_value+4*i);

   for (i=0;i<32;i++)
      msg_u32[i]=U8TO32_LITTLE(message_block+4*i);

   for (i=0;i<16;i++)
      salt_u32[i]=U8TO32_LITTLE(salt+4*i);

   cnt[1]=(uint32_t)(counter>>32);
   cnt[0]=(uint32_t)(counter & 0xFFFFFFFFULL);

/* Due to NIST specifications, there are no messages of length       */
/* longer than 2^64 that are expected				     */

   cnt[2]=0;
   cnt[3]=0;

/* Computing the encryption function                                 */

   E512(pt, ct, msg_u32, salt_u32, cnt);


/* Davies-Meyer transformation                                       */
 
   for (i=0;i<16;i++) 
       pt[i]^=ct[i]; 

/* Translating the output to 8-bit words                             */

   for(i=0; i<16; i++)
       U32TO8_LITTLE(chaining_value+i*4, pt[i]); 

   return;
}
/* ===== */

/* Initialization of the internal state of the hash function         */

HashReturn Init(hashState *state, int hashbitlen)
{

/* Check that the requested digest length is not negative, not zero, */
/* and not more than 512 bits                                        */

   if (hashbitlen>512) return BAD_HASHBITLEN;
   if (hashbitlen<1) return BAD_HASHBITLEN;


/* Setting the salt to zero. Applications which wish to use          */
/* different salt should initialize it after (!) the derivation of   */
/* the IVs                                                           */

   memset(state->salt,0,64);


/* Initialization of the counter of number of bits that were hashed  */
/* so far                                                            */ 

   state->bitcount = 0;


/* Store the requested digest size                                   */

   state->DigestSize = hashbitlen;


/* Initialize the message block to empty                             */

   memset(state->buffer,0,128);


/* Set the input to the compression function to all zero             */

   memset(state->chaining_value,0,64); 

   
/* Compute the respective MIV and the respective IV. These values    */
/* depend on the used compression function.                          */

   if (hashbitlen<257)
      {

/* Compute MIV_{256}                                                 */

         Compress256(state->buffer,state->chaining_value,0x0ULL,state->salt);
         

/* Set the message block to the size of the requested digest size    */
         
         U16TO8_LITTLE(state->buffer,hashbitlen);


/* Compute IV_m                                                      */

         Compress256(state->buffer,state->chaining_value,0x0ULL,state->salt);

/* Set the block size to be 512 bits (as required for C_{256})       */
        
         state->BlockSize = 512;
      }

   else

      {

/* Compute MIV_{512}                                                 */

         Compress512(state->buffer,state->chaining_value,0x0UL,state->salt);

/* Set the message block to the size of the requested digest size    */
         
         U16TO8_LITTLE(state->buffer,hashbitlen);


/* Compute IV_m                                                      */

         Compress512(state->buffer,state->chaining_value,0x0ULL,state->salt);

/* Set the block size to be 1024 bits (as required for C_{512})      */
 
         state->BlockSize=1024; 

     }


/* Set the message block to zero				     */

   memset(state->buffer,0,128);


   return SUCCESS;
}



/* Compressing the input data, and updating the internal state       */

HashReturn Update (hashState *state, const BitSequence *data, DataLength
                   databitlen)
{

/* p is a pointer to the current location inside data that we need   */
/* to process (i.e., the first byte of the data which was not used   */
/* as an input to the compression function                           */
   uint8_t *p = (uint8_t*)data;

/* len is the size of the data that was not process yet in bytes     */
   int len = databitlen>>3;

/* BlockSizeB is the size of the message block of the compression    */
/* function                                                          */
   int BlockSizeB = (state->BlockSize/8);

/* bufcnt stores the number of bytes that are were "sent" to the     */
/* compression function, but were not yet processed, as a full block */
/* has not been obtained                                             */
   int bufcnt= (state->bitcount>>3)%BlockSizeB;

/* local_bitcount contains the number of bits actually hashed so far */
   uint64_t local_bitcount;

/* If we had to process a message with partial bytes before, then    */
/* Update() should not have been called again.                       */
/* We just discard the extra bits, and inform the user               */

   if (state->bitcount&7ULL)
      {
         fprintf(stderr, "We are sorry, you are calling Update one time after\n"
 		         "what should have been the last call. We ignore\n"
		         "few bits of the input.\n");
         state->bitcount &= ~7ULL;
      }


/* load the number of bits hashed so far into local_bitcount         */

   local_bitcount=state->bitcount;


/* mark that we processed more bits                                  */

   state->bitcount += databitlen;


/* if the input contains a partial byte - store it independently     */

   if (databitlen&7)
      state->partial_byte = data[databitlen>>3];


/* Check if we have enough data to call the compression function     */
/* If not, just copy the input to the buffer of the message block    */

   if (bufcnt + len < BlockSizeB)
      {
         memcpy(&state->buffer[bufcnt], p, len);
         return SUCCESS;
      }


/* There is enough data to start calling the compression function.   */
/* We first check whether there is data remaining from previous      */
/* calls                                                             */

   if (bufcnt>0)
      {

/* Copy from the input the required number of bytes to fill a block  */

         memcpy(&state->buffer[bufcnt], p, BlockSizeB-bufcnt);

/* Update the location of the first byte that was not processed      */

         p += BlockSizeB-bufcnt;

/* Update the remaining number of bytes to process                   */

         len -= BlockSizeB-bufcnt;

/* Update the number of bits hashed so far (locally)                 */

         local_bitcount+=8*(BlockSizeB-bufcnt);

/* Call the respective compression function to process the current   */
/* block                                                             */

         if (state->DigestSize<257)
            Compress256(state->buffer, state->chaining_value, local_bitcount, state->salt);
         else
            Compress512(state->buffer, state->chaining_value, local_bitcount, state->salt);
    }


/* At this point, the only remaining data is from the message block  */
/* call the compression function as many times as possible, and      */
/* store the remaining bytes in the buffer                           */

/* Each step of the loop compresses BlockSizeB bytes                 */

   for( ; len>=BlockSizeB; len-=BlockSizeB, p+=BlockSizeB)
      {

/* Update the number of bits hashed so far (locally)                 */

         local_bitcount+=BlockSizeB;

/* Call the respective compression function to process the current   */
/* block                                                             */

         if (state->DigestSize<257)
            Compress256(p, state->chaining_value, local_bitcount, state->salt);
         else 
            Compress512(p, state->chaining_value, local_bitcount, state->salt);

      }


/* If there are still unprocessed bytes, store them locally and wait */
/* for more                                                          */

   if (len>0)
      memcpy(state->buffer, p, len);

   return SUCCESS;
}


/* Performing the padding scheme, and dealing with any remaining     */
/* bits                                                              */

HashReturn Final (hashState *state, BitSequence *hashval)
{

/* Stores inputs (message blocks) to the compression function        */

   uint8_t block[128];

/* Stores results (chaining value) of the compression function       */

   uint8_t result[64];

/* BlockSizeB is the size of the message block of the compression    */
/* function                                                          */

   int BlockSizeB = (state->BlockSize/8);

/* bufcnt stores the number of bytes that are were "sent" to the     */
/* compression function, but were not yet processed, as a full block */
/* has not been obtained                                             */

   int bufcnt= ((uint64_t)state->bitcount>>3)%BlockSizeB;

   int i;

/* Copy the current chaining value into result (as a temporary step) */

   if (state->DigestSize < 257)
      memcpy(result, state->chaining_value, 32);
   else
      memcpy(result, state->chaining_value, 64);


/* Initialize block as the message block to compress with the bytes  */
/* that were not processed yet                                       */

   memset(block, 0, BlockSizeB);
   memcpy(block, state->buffer, bufcnt);


/* Pad the buffer with the byte which contains the fraction of bytes */
/* from and a bit equal to 1					     */

   block[bufcnt] = (state->partial_byte&\
      ~((0x80 >> (state->bitcount&7))-1)) | (0x80 >> (state->bitcount&7));


/* Compress the last block (according to the digest size)            */

  if (state->DigestSize<257) {


/* An additional message block is required if there are less than 10 */
/* more bytes for message length and digest length encoding          */

     if (bufcnt>=BlockSizeB-10)
        {

/* Compress the current block                                        */
           Compress256(block,result,state->bitcount,state->salt);

/* Generate the full padding block                                   */
           memset(block, 0, BlockSizeB);
           U64TO8_LITTLE(block+BlockSizeB-10, state->bitcount);
           U16TO8_LITTLE(block+BlockSizeB-2, state->DigestSize);

/* Compress the full padding block                                   */
           Compress256(block,result,0x0ULL,state->salt);

        }

     else

        {

/* Pad the number of bits hashed so far and the digest size to the  */
/* last message block and compress it				    */
           U64TO8_LITTLE(block+BlockSizeB-10, state->bitcount);
           U16TO8_LITTLE(block+BlockSizeB-2, state->DigestSize);
           Compress256(block,result, state->bitcount, state->salt);

        }
   }

/* In case the requested digest size is more than 256 bits           */ 

   else

      {

/* An additional message block is required if there are less than 18 */
/* more bytes for message length and digest length encoding          */
         if (bufcnt>=BlockSizeB-18)
            {

/* Compress the current block                                        */
               Compress512(block,result,state->bitcount,state->salt);

/* Generate the full padding block                                   */
               memset(block, 0, BlockSizeB);
               U64TO8_LITTLE(block+BlockSizeB-18, state->bitcount);
               U16TO8_LITTLE(block+BlockSizeB-2, state->DigestSize);

/* Compress the full padding block                                   */
               Compress512(block,result,0x0UL,state->salt);
            }

         else

            {
/* Pad the number of bits hashed so far and the digest size to the   */
/* last message block and compress it				     */
               U64TO8_LITTLE(block+BlockSizeB-18, state->bitcount);
               memset(block+BlockSizeB-10,0,8);
               U16TO8_LITTLE(block+BlockSizeB-2, state->DigestSize);
               Compress512(block,result, state->bitcount, state->salt);

            }
      }


/* Copy the result into the supplied array of bytes.                 */

   for (i=0;i<(state->DigestSize+7)/8;i++)
       hashval[i]=result[i];


/* Treat cases where the digest size is not a multiple of a byte     */

   if ((state->DigestSize)&7)
       hashval[(state->DigestSize+7)/8] &= \
              (0xFF<<(8-((state->DigestSize)%8)))&0xFF;

   return SUCCESS;

}


/* Hashing a message, from initialization till the end               */

HashReturn Hash (int hashbitlen, const BitSequence *data,
                 DataLength databitlen, BitSequence *hashval)
{
   HashReturn Status;
   hashState state;

   Status = Init(&state, hashbitlen);
   if (Status != SUCCESS) return (Status);
   Status = Update (&state, data, databitlen);
   if (Status != SUCCESS) return (Status);
   Status = Final (&state, hashval);
   if (Status != SUCCESS) return (Status);

   return (SUCCESS);

}
