/*

Copyright 2015, Mentor Graphics Corporation
http://www.mentor.com

*/

#include <iostream>
#include "Poco/Net/HTTPRequest.h"
#include "Poco/Net/HTTPResponse.h"
#include "Poco/Timespan.h"
#include "Poco/Timestamp.h"
#include "Poco/Exception.h"
#include "Poco/JSON/Parser.h"
#include "Poco/Dynamic/Var.h"
#include "Poco/JSON/Object.h"
#include "Poco/Net/NetException.h"
#include "signalRClient/Transport.h"
#include "signalRClient/WebSocketsTransport.h"
#include "signalRClient/Connection.h"
#include "signalRClient/ConnectionLife.h"
#include "signalRClient/ClientSession.h"
#include "signalRClient/SignalRException.h"

using std::endl;
using std::istringstream;
using std::stringstream;
using Poco::Net::HTTPRequest;
using Poco::Net::HTTPResponse;
using Poco::Net::HTTPMessage;
using Poco::Timespan;
using Poco::Timestamp;
using Poco::TimeoutException;
using Poco::Dynamic::Var;
using namespace Poco::JSON;
using Poco::Net::NetException;
using Poco::Net::ConnectionResetException;

namespace signalRClient  {

WebSocketsTransport::WebSocketsTransport(Connection* pConnection)
: Transport(PROTOCOL_WEBSOCKETS, pConnection)
, pWebSocket(0)
, pConnectionLife(0) 
, clientClose(false) {

}

WebSocketsTransport::~WebSocketsTransport() {

    if (asyncReceive.isRunning()) {
        asyncReceive.stop();
        asyncReceive.wait();
    }

    if (pWebSocket) delete pWebSocket;
    pWebSocket = 0;
}

bool WebSocketsTransport::doConnect() {

    try {
        Timespan tct((pConnection->transportConnectTimeout)*TO_MICROSECOND);

        ClientSession cs(pConnection->host, pConnection->port, pConnection->secure);

        cs.getSession()->setTimeout(tct);

        const string& connectQ = createQuery(pConnection, qConnect);
        pConnection->diagnosticLogger.post(DIAG_DEBUG) << "[WebSockets] Issuing query: " << connectQ << endl;

        HTTPRequest connRequest(HTTPRequest::HTTP_GET, connectQ, HTTPMessage::HTTP_1_1);
        HTTPResponse connResponse;

        pWebSocket = new WebSocket(*(cs.getSession()), connRequest, connResponse);

        if (pWebSocket) {
            pConnection->diagnosticLogger.post(DIAG_DEBUG) << "[WebSockets] /connect is successful" << endl;
            //pWebSocket->setReceiveTimeout(remaining);
            pWebSocket->setReceiveTimeout(0);

            char frame[256];
            int flags = 0;
            int l = pWebSocket->receiveFrame(frame, 256, flags);

            if( l < 256 ) {
                frame[l] = 0;
                pConnection->diagnosticLogger.post(DIAG_DEBUG) << "[WebSockets] First frame: " << frame << endl;
            } else {
                pConnection->diagnosticLogger.post(DIAG_DEBUG) << "[WebSockets] Cannot print as buffer is Full" << frame << endl;
            }
            istringstream str(string(frame, l));
            processQueryResponse(pConnection, qConnect, str);
            pWebSocket->setReceiveTimeout(0);

            if (messageId.empty() || !initialized || message.size()) {
                pConnection->diagnosticLogger.post(DIAG_ERROR) << "[WebSockets] Unable to connect: First WebSocket frame is invalid, frame: " << frame << endl;
                return false;
            }
        } else {
            pConnection->diagnosticLogger.post(DIAG_ERROR) << "[WebSockets] Unable to connect: Websocket is not initialized" << endl;
            return false;
        }

        if (pConnection->stopped) return false;

        pConnectionLife = new ConnectionLife(pConnection);

        asyncReceive.start();

        pConnectionLife->start();
    
    } catch (Poco::Exception& pocoEx) {
        pConnection->diagnosticLogger.post(DIAG_ERROR) << "[WebSockets] Unable to connect: " << pocoEx.displayText() << endl; 
        return false;
    }

    return true;
}

void WebSocketsTransport::doSend(const string& message) {

    bool err = false;
    if (pWebSocket) {
        try {
            int sentsize = pWebSocket->sendFrame(message.c_str(), message.size(), WebSocket::FRAME_TEXT);
            if (sentsize < 0) {
                pConnection->diagnosticLogger.post(DIAG_ERROR) << "[WebSockets] Unable to send message: socket send error code " << sentsize << endl;
                err = true;
            }
            if (sentsize < message.size()){
                pConnection->diagnosticLogger.post(DIAG_ERROR) << "[WebSockets] Unable to send full message: socket send buffer already full" << endl;
                err = true;
            }
        } catch (Poco::Exception& pocoEx) {
            pConnection->diagnosticLogger.post(DIAG_ERROR) << "[WebSockets] [WebSockets] Unhandled exception in WebSocketsTransport::doSend: " << pocoEx.displayText() << endl;
            err = true;
        }
    } else {
        pConnection->diagnosticLogger.post(DIAG_ERROR) << "[WebSockets] Unable to send message: Websocket is not connected" << endl;
        err = true;
    }

    if (err) 
        pConnection->notificationCenter.postNotification(new ErrorEvent("Unable to send message"));
}

void WebSocketsTransport::doReceive() {

    bool err = false;
    try {
        pConnection->diagnosticLogger.post(DIAG_DEBUG) << "[WebSockets] Starting receive loop" << endl;
        ServerTerminationType close = stNone;
        pWebSocket->setReceiveTimeout(pConnection->getKeepAliveTimeout()*TO_MICROSECOND);

        while (!asyncReceive.isStopped()) {
            string mesg;
            try {
                mesg = assembleFrames(close);
            } catch (TimeoutException& timeoutEx) {
                if (clientClose) {
                    pConnection->diagnosticLogger.post(DIAG_INFORMATION) << "[WebSockets] Client has closed the connection" << endl;
                    break;
                } else 
                    continue;
            }

            if (close) {
                break;
            }

            try {
                if (isKeepAlive(mesg)) {
                    pConnection->notificationCenter.postNotification(new KeepAliveEvent());
                    pWebSocket->setReceiveTimeout(pConnection->getKeepAliveTimeout()*TO_MICROSECOND);
                } else {
                    istringstream str(mesg);
                    processDataReceived(this, str);
                    for (int i = 0; i < message.size(); i++) {

                        string msg;
                       if (message.isObject(i)) {
                            stringstream outStr;
                            Object::Ptr object = message.getObject(i);
                            object->stringify(outStr);
                            msg = outStr.str();
                        } else {
                            msg = message.getElement<string>(i);
                        }
                        pConnection->notificationCenter.postNotification(new MessageReceivedEvent(msg));
                    }
                }
            } catch (JSONException& jsonEx) {
                pConnection->diagnosticLogger.post(DIAG_WARNING) << "[WebSockets] Invalid JSON message received: " << jsonEx.displayText() << endl;
            }
        }

        if (!clientClose && close) { //server initiated close
            asyncReconnect(close);
        }
    } catch (Poco::Exception& pocoEx) {
        pConnection->diagnosticLogger.post(DIAG_ERROR) << "[WebSockets] Unhandled exception in WebSocketsTransport::doReceive: " << pocoEx.displayText() << endl;
        err = true;
    } catch (std::exception& stdEx) {
        pConnection->diagnosticLogger.post(DIAG_ERROR) << "[WebSockets] Unhandled std::exception in WebSocketsTransport::doReceive: " << stdEx.what() << endl;
        err = true;
    }

    if (err)
        pConnection->notificationCenter.postNotification(new ErrorEvent("Unable to receive"));
}

string WebSocketsTransport::assembleFrames(ServerTerminationType& close) {

    char* buff = 0;
    string mesg;

    try {
        int buffsize = pWebSocket->getReceiveBufferSize();
        //pConnection->diagnosticLogger.post(DIAG_DEBUG) << "[WebSockets] Allocating " << buffsize << " bytes to receive frame" << endl;
        buff = new char[buffsize+1];

        bool loop = true;
        bool start = false;

        int messagesize = 0;

        while (loop) {
            int flags = 0;
            int length = 0;
            try {
                length = pWebSocket->receiveFrame(buff, buffsize, flags);
            } catch (TimeoutException& timeoutEx) {
                pConnection->diagnosticLogger.post(DIAG_DEBUG) << "[WebSockets] Timed out while receiving frame" << endl; 
                delete[] buff;
                throw timeoutEx;
            } catch (NetException& netEx) {
                pConnection->diagnosticLogger.post(DIAG_ERROR) << "[WebSockets] Unable to receive frame: " << netEx.displayText() << endl; 
                loop = false;
                close = stAbnormal;
                break;
            }

            buff[length] = 0;

            pConnection->diagnosticLogger.post(DIAG_DEBUG) << "[WebSockets] Frame flags: " << std::hex << flags << std::dec << endl;
            if ((flags & WebSocket::FRAME_OP_CLOSE) == WebSocket::FRAME_OP_CLOSE) {
                loop = false;
                close = stNormal;
                break;
            }

            if (length) {
                pConnection->diagnosticLogger.post(DIAG_DEBUG) << "[WebSockets] Frame length: " << length << endl;

                if ((flags & WebSocket::FRAME_OP_TEXT) == WebSocket::FRAME_OP_TEXT) {
                    start = true;
                } else if (!((flags & WebSocket::FRAME_OP_CONT) == WebSocket::FRAME_OP_CONT)) {
                    pConnection->diagnosticLogger.post(DIAG_WARNING) << "[WebSockets] Frame ignored" << endl;
                }

                if (start) {
                    mesg.append(buff);
                    messagesize += length;
                }

                if ((flags & WebSocket::FRAME_FLAG_FIN) == WebSocket::FRAME_FLAG_FIN) {
                   loop = false;
                   start = false;
                }
            } else {
                pConnection->diagnosticLogger.post(DIAG_ERROR) << "[WebSockets] Websocket is closed; length is 0" << endl;
                loop = false;
                close = stAbnormal;
            }
        }

        delete[] buff;

    } catch (TimeoutException& timeoutEx) {
        throw timeoutEx;
    } catch (Poco::Exception& pocoEx) {
        pConnection->diagnosticLogger.post(DIAG_ERROR) << "[WebSockets] Unhandled exception in WebSocketsTransport::assembleFrames: " << pocoEx.displayText() << endl;
        if (buff) delete[] buff;
        throw pocoEx;
    } catch (std::exception& stdEx) {
        pConnection->diagnosticLogger.post(DIAG_ERROR) << "[WebSockets] Unhandled std::exception in WebSocketsTransport::assembleFrames: " << stdEx.what() << endl;
        if (buff) delete[] buff;
        throw stdEx;
    }

    return mesg;
}

bool WebSocketsTransport::isKeepAlive(const string& mesg) {

    Parser parser;
    Var result = parser.parse(mesg);
    Object::Ptr object = result.extract<Object::Ptr>();

    if(object->size() == 0) {
        return true;
    }

    return false;
}

void WebSocketsTransport::doDisconnect() {

    bool err = false;
    try {
        if (pConnectionLife) delete pConnectionLife;
        pConnectionLife = 0;

        pConnection->diagnosticLogger.post(DIAG_DEBUG) << "[WebSockets] Shutting down socket" << endl;
        pWebSocket->shutdown();
        clientClose = true;

        pConnection->diagnosticLogger.post(DIAG_DEBUG) << "[WebSockets] Stopping receive loop" << endl;
        if (asyncReceive.isRunning()) {
            asyncReceive.stop();
            asyncReceive.wait();
        }

        pConnection->diagnosticLogger.post(DIAG_DEBUG) << "[WebSockets] Closing the socket" << endl;
        pWebSocket->close();

    } catch (Poco::Exception& pocoEx) {
        pConnection->diagnosticLogger.post(DIAG_ERROR) << "[WebSockets] Unhandled exception in WebSocketsTransport::doDisconnect: " << pocoEx.displayText() << endl;
        err = true;        
    } catch (std::exception& stdEx) {
        pConnection->diagnosticLogger.post(DIAG_ERROR) << "[WebSockets] Unhandled std::exception in WebSocketsTransport::doDisconnect: " << stdEx.what() << endl;
        err = true;
    }

    if (pWebSocket) delete pWebSocket;
    pWebSocket = 0;

    if (err) {
        throw SignalRException("Unable to disconnect");
    }

    pConnection->doAbort();
}

void WebSocketsTransport::doReconnect(const int& serverInitiatedClose) {

    bool err = false;
    if (!(pConnection->syncDisconnectLock.tryReadLock())) {
        pConnection->diagnosticLogger.post(DIAG_DEBUG) << "[WebSockets] Unable to reconnect: Could not acquire lock; disconnection in progress" << endl;
        return;
    }
    pConnection->syncLock.lock();

    try {
        pConnection->diagnosticLogger.post(DIAG_INFORMATION) << "[WebSockets] Reconnecting" << endl;

        pConnection->setState(sReconnecting);
        pConnection->notificationCenter.postNotification(new ReconnectingEvent());

        if (pConnectionLife) delete pConnectionLife;
        pConnectionLife = 0;

        if (serverInitiatedClose == stNormal) {
            pConnection->diagnosticLogger.post(DIAG_DEBUG) << "[WebSockets] Shutting down socket" << endl;
            pWebSocket->shutdown();
        }

        if (asyncReceive.isRunning()) {
            pConnection->diagnosticLogger.post(DIAG_DEBUG) << "[WebSockets] Stopping receive loop" << endl;
            asyncReceive.stop();
            asyncReceive.wait();
        }

        pConnection->diagnosticLogger.post(DIAG_DEBUG) << "[WebSockets] Closing the socket" << endl;
        pWebSocket->close();
        delete pWebSocket;
        pWebSocket = 0;

        bool reconnected = false;

        Timespan remainingTime((pConnection->disconnectTimeout)*TO_MICROSECOND);
        Timestamp start;
        const string& reconnectQ = createQuery(pConnection, qReconnect);
        pConnection->diagnosticLogger.post(DIAG_DEBUG) << "[WebSockets] Issuing query: " << reconnectQ << endl;

        bool msg = false;
        while (!reconnected) {
       
            try {
                start.update();
                ClientSession cs(pConnection->host, pConnection->port, pConnection->secure);
                HTTPRequest connRequest(HTTPRequest::HTTP_GET, reconnectQ, HTTPMessage::HTTP_1_1);
                HTTPResponse connResponse;
                cs.getSession()->setTimeout(remainingTime);
                pWebSocket = new WebSocket(*(cs.getSession()), connRequest, connResponse);
    
            } catch (NetException& netEx) {

                if (pConnection->stopped) break;

                Timestamp end;
                Timestamp::TimeDiff elapsed = end - start;
                remainingTime = remainingTime - elapsed;
                if (remainingTime.totalMicroseconds() <= 0) {
                    pConnection->diagnosticLogger.post(DIAG_ERROR) << "[WebSockets] Unable to reconnect: timed out, given disconnect timeout: " << pConnection->disconnectTimeout << endl;
                    break;
                }
                if (!msg) { 
                    pConnection->diagnosticLogger.post(DIAG_DEBUG) << "[WebSockets] Trying to reconnect again" << endl;
                    msg = true;
                }
                continue;
            } catch (TimeoutException& timeoutEx) {
                pConnection->diagnosticLogger.post(DIAG_ERROR) << "[WebSockets] Unable to reconnect: timed out, given disconnect timeout: " << pConnection->disconnectTimeout << "; " << timeoutEx.displayText() << endl;
                break;
            }

            if (pWebSocket) 
                reconnected = true;
        }

        if (reconnected) {
            pConnection->diagnosticLogger.post(DIAG_DEBUG) << "[WebSockets] /reconnect is successful" << endl;
            pConnectionLife = new ConnectionLife(pConnection);

            asyncReceive.start();

            pConnectionLife->start();

            pConnection->setState(sConnected);
            pConnection->notificationCenter.postNotification(new ReconnectedEvent());
            pConnection->diagnosticLogger.post(DIAG_INFORMATION) << "[WebSockets] Reconnected" << endl;
        } else {
            //do not need /abort after disconnect timeout
            pConnection->setState(sDisconnected);
            pConnection->notificationCenter.postNotification(new DisconnectedEvent());
            pConnection->diagnosticLogger.post(DIAG_INFORMATION) << "[WebSockets] Disconnected after trying to reconnect" << endl;
            err = true;
        }

    } catch (Poco::Exception& pocoEx) {
        pConnection->diagnosticLogger.post(DIAG_ERROR) << "[WebSockets] Unhandled exception in WebSocketsTransport::doReconnect: " << pocoEx.displayText() << endl;
        if (pWebSocket) delete pWebSocket;
        pWebSocket = 0;
        err = true;
    } catch (std::exception& stdEx) {
        pConnection->diagnosticLogger.post(DIAG_ERROR) << "[WebSockets] Unhandled std::exception in WebSocketsTransport::doReconnect: " << stdEx.what() << endl;
        if (pWebSocket) delete pWebSocket;
        pWebSocket = 0;
        err = true;
    }

    if (err)
        pConnection->notificationCenter.postNotification(new ErrorEvent("Unable to reconnect"));

    pConnection->syncLock.unlock();
    pConnection->syncDisconnectLock.unlock();
}

}
