/*

Copyright 2015, Mentor Graphics Corporation
http://www.mentor.com

*/

#include "Poco/JSON/Parser.h"
#include "Poco/Dynamic/Var.h"
#include "Poco/JSON/Object.h"
#include "Poco/URI.h"
#include "signalRClient/Util.h"
#include "signalRClient/Connection.h"

using Poco::Dynamic::Var;
using namespace Poco::JSON;
using Poco::URI;

#define ENCODING_RESERVED_SET ":/?#[]@!$&()*+,;="
//#include <iostream>
//using namespace std;

namespace signalRClient {

string createQuery(const Connection* pConnection, const QueryType& queryType) {
    
    string query;

    try {

    query += pConnection->baseUrl + "/";

    switch(queryType) {
        case qNegotiate:
        {
            query += "negotiate?";
            query += "clientProtocol=" + pConnection->protocolVersion;
            if(!pConnection->customQuery.empty())
                query += "&" + pConnection->customQuery;
        }
        break;
        case qConnect:
        {
            query += "connect?";
            query += "transport=" + pConnection->getTransport()->getName();
            query += "&clientProtocol=" + pConnection->protocolVersion;

            string encodedconntoken;
            URI::encode(pConnection->connectionToken, ENCODING_RESERVED_SET, encodedconntoken);

            query += "&connectionToken=" + encodedconntoken;
            if(!pConnection->customQuery.empty())
                query += "&" + pConnection->customQuery;
        }
        break;
        case qStart:
        {
            query += "start?";
            query += "transport=" + pConnection->getTransport()->getName();
            query += "&clientProtocol=" + pConnection->protocolVersion;

            string encodedconntoken;
            URI::encode(pConnection->connectionToken, ENCODING_RESERVED_SET, encodedconntoken);

            query += "&connectionToken=" + encodedconntoken;
            if(!pConnection->customQuery.empty())
                query += "&" + pConnection->customQuery;
        }
        break;
        case qSend:
        {
            query += "send?";
            query += "transport=" + pConnection->getTransport()->getName();
            query += "&clientProtocol=" + pConnection->protocolVersion;

            string encodedconntoken;
            URI::encode(pConnection->connectionToken, ENCODING_RESERVED_SET, encodedconntoken);

            query += "&connectionToken=" + encodedconntoken;
            if(!pConnection->customQuery.empty())
                query += "&" + pConnection->customQuery;
        }
        break;
        case qPing:
        break;
        case qAbort:
        {
            query += "abort?";
            query += "transport=" + pConnection->getTransport()->getName();
            query += "&clientProtocol=" + pConnection->protocolVersion;

            string encodedconntoken;
            URI::encode(pConnection->connectionToken, ENCODING_RESERVED_SET, encodedconntoken);

            query += "&connectionToken=" + encodedconntoken;
            if(!pConnection->customQuery.empty())
                query += "&" + pConnection->customQuery;
        }
        break;
        case qPoll:
        {
            query += "poll?";
            query += "transport=" + pConnection->getTransport()->getName();
            query += "&clientProtocol=" + pConnection->protocolVersion;

            string encodedconntoken;
            URI::encode(pConnection->connectionToken, ENCODING_RESERVED_SET, encodedconntoken);

            query += "&connectionToken=" + encodedconntoken;

            string encodedMessageId;
            URI::encode(pConnection->pTransport->messageId, ":/?#[]@!$&()*+,;=", encodedMessageId);

            query += "&messageId=" + encodedMessageId;
            if(!pConnection->customQuery.empty())
                query += "&" + pConnection->customQuery;
        }
        break;
        case qReconnect:
        {
            query += "reconnect?";
            query += "transport=" + pConnection->getTransport()->getName();
            query += "&clientProtocol=" + pConnection->protocolVersion;

            string encodedconntoken;
            URI::encode(pConnection->connectionToken, ENCODING_RESERVED_SET, encodedconntoken);

            query += "&connectionToken=" + encodedconntoken;
            if(!pConnection->customQuery.empty())
                query += "&" + pConnection->customQuery;
        }
        break;
        default:
            throw Poco::Exception("<queryType> is unknown");
    }

    } catch (Poco::Exception& pocoEx) {
        throw Poco::Exception("exception in createQuery: " + pocoEx.displayText());
    } catch (std::exception& stdEx) {
        throw Poco::Exception("std::exception in createQuery: " + string(stdEx.what()));
    }

    //cout << "[Negotation] query = " << query << endl;
    return query;
}

void processQueryResponse(Connection* pConnection, const QueryType& queryType, istream& data) {

    try {

        Parser parser;
        Var result = parser.parse(data);
        Object::Ptr object = result.extract<Object::Ptr>();

        switch(queryType) {
            case qNegotiate:
            {
                Var url = object->get("Url");
                pConnection->baseUrl = url.convert<string>();
                //cout << "[Negotation] Url = " << pConnection->baseUrl << endl;

                Var conntoken = object->get("ConnectionToken");
                pConnection->connectionToken = conntoken.convert<string>();
                //cout << "[Negotation] ConnectionToken = " << pConnection->connectionToken << endl;

                Var connid = object->get("ConnectionId");
                pConnection->connectionId = connid.convert<string>();
                //cout << "[Negotation] Connection Id = " << pConnection->connectionId << endl;

                Var conntimeout = object->get("ConnectionTimeout");
                pConnection->connectionTimeout = conntimeout.convert<long>();
                //cout << "[Negotation] ConnectionTimeout = " << pConnection->connectionTimeout << endl;

                Var katimeout = object->get("KeepAliveTimeout");
                pConnection->keepAliveTimeout = katimeout.convert<long>();
                //cout << "[Negotation] KeepAliveTimeout = " << pConnection->keepAliveTimeout << endl;

                Var disconntimeout = object->get("DisconnectTimeout");
                pConnection->disconnectTimeout = disconntimeout.convert<long>();
                //cout << "[Negotation] DisconnectTimeout = " << pConnection->disconnectTimeout << endl;

                Var trywebsockets = object->get("TryWebSockets");
                pConnection->tryWebSockets = trywebsockets.convert<bool>();
                //cout << "[Negotation] TryWebSockets = " << pConnection->tryWebSockets << endl;

                Var protocolver = object->get("ProtocolVersion");
                pConnection->protocolVersion = protocolver.convert<string>();
                //cout << "[Negotation] ProtocolVersion = " << pConnection->protocolVersion << endl;

                Var tctimeout = object->get("TransportConnectTimeout");
                pConnection->transportConnectTimeout = tctimeout.convert<long>();
                //cout << "[Negotation] TransportConnectTimeout = " << pConnection->transportConnectTimeout << endl;

                Var lpdelay = object->get("LongPollDelay");
                pConnection->longPollDelay = lpdelay.convert<long>();
                //cout << "[Negotation] LongPollDelay = " << pConnection->longPollDelay << endl;
            }
            break;
            case qConnect:
            {
                Var messageid = object->get("C");
                pConnection->pTransport->messageId = messageid.convert<string>();
                //cout << "[Transport] C = " << pConnection->getTransport()->messageId << endl;

                Var initialized = object->get("S");
                pConnection->pTransport->initialized = initialized.convert<long>();
                //cout << "[Transport] S = " << pConnection->getTransport()->initialized << endl;

                //cout << "[Transport] M size = " << object->getArray("M")->size()<< endl;
                if(!object->getArray("M")->size())
                    pConnection->pTransport->message = *object->getArray("M");
                if (pConnection->pTransport->message.size() != 0) {
                    //error: connect request returned a non-empty message
                }
            }
            break;
            case qStart:
            {
                Var response = object->get("Response");
                pConnection->startResponse = response.convert<string>();        
                //cout << "[Transport] Response = " << pConnection->startResponse << endl;
            }
            break;
            case qSend:
            break;
            case qPing:
            break;
            case qAbort:
            break;
            case qPoll:
            {
                Var messageid = object->get("C");
                pConnection->pTransport->messageId = messageid.convert<string>();
                //cout << "[Transport] C = " << pConnection->getTransport()->messageId << endl;

                //cout << "[Transport] M size = " << object->getArray("M")->size()<< endl;
                pConnection->pTransport->message = *object->getArray("M");
            }
            break;
            case qReconnect:
            {
                Var messageid = object->get("C");
                pConnection->pTransport->messageId = messageid.convert<string>();
                //cout << "[Transport] C = " << pConnection->getTransport()->messageId << endl;

                //cout << "[Transport] M size = " << object->getArray("M")->size()<< endl;
                pConnection->pTransport->message = *object->getArray("M");
            }
            break;
            default:
                throw Poco::Exception("<queryType> is unknown");
        }

    } catch (Poco::Exception& pocoEx) {
        throw Poco::Exception("exception in processQueryResponse: " + pocoEx.displayText());
    } catch (std::exception& stdEx) {
        throw Poco::Exception("std::exception in processQueryResponse: " + string(stdEx.what()));
    }
}

void processDataReceived(Transport* pTransport, istream& data) {

    try {

        Parser parser;
        Var result = parser.parse(data);
        Object::Ptr object = result.extract<Object::Ptr>();

        Var messageid = object->get("C");
        pTransport->messageId = messageid.convert<string>();
        //cout << "[Transport - received] C = " << pTransport->messageId << endl;

        //cout << "[Transport - received] M size = " << object->getArray("M")->size()<< endl;
        if(object->getArray("M")->size()) {
            pTransport->message = *object->getArray("M");
            //stringstream outStr;
            //pTransport->message.stringify(outStr);
            //cout << "[Transport - received] M = " << outStr.str() << endl;
        }

    } catch (Poco::Exception& pocoEx) {
        throw Poco::Exception("exception in processDataReceived: " + pocoEx.displayText());
    } catch (std::exception& stdEx) {
        throw Poco::Exception("std::exception in processDataReceived: " + string(stdEx.what()));
    }
}

}
