/*

Copyright 2015, Mentor Graphics Corporation
http://www.mentor.com

*/

#include "signalRClient/Transport.h"
#include "signalRClient/LongPollingTransport.h"
#include "signalRClient/Connection.h"
#include "signalRClient/ClientSession.h"

#include "Poco/Net/HTTPRequest.h"
#include "Poco/Net/HTTPResponse.h"
#include "Poco/Net/HTTPMessage.h"
#include "Poco/Net/HTMLForm.h"
#include "Poco/StreamCopier.h"
#include "Poco/URI.h"
#include "Poco/Timespan.h"
#include "Poco/Exception.h"
#include "Poco/Stopwatch.h"

using Poco::Net::HTTPRequest;
using Poco::Net::HTTPResponse;
using Poco::Net::HTTPMessage;
using Poco::Net::HTMLForm;
using Poco::StreamCopier;
using Poco::URI;
using Poco::Timespan;
using Poco::TimeoutException;
using Poco::Stopwatch;

using namespace std;


namespace signalRClient  {

LongPollingTransport::LongPollingTransport(Connection* pConnection)
: Transport(PROTOCOL_LONGPOLLING, pConnection), logger(this->pConnection->diagnosticLogger) {
}

LongPollingTransport::~LongPollingTransport() {
}

bool LongPollingTransport::doConnect() {
    bool result=false;
    ClientSession connectSession(pConnection->host, pConnection->port, pConnection->secure);
    connectSession.getSession()->setTimeout(Timespan(pConnection->transportConnectTimeout*1000*1000));
    connectSession.getSession()->setKeepAlive(true);

    logger.post(DIAG_DEBUG) << "entering -- " << __func__ << endl;
    try {
        HTTPRequest request(HTTPRequest::HTTP_GET, createQuery(this->pConnection, qConnect), HTTPMessage::HTTP_1_1);
        connectSession.getSession()->sendRequest(request);

        //request.write(cout);

        HTTPResponse response;
        istream& data = connectSession.getSession()->receiveResponse(response);
        HTTPResponse::HTTPStatus status = response.getStatus();
        if (status != HTTPResponse::HTTP_OK) {
            //error: bad response
            logger.post(DIAG_ERROR) << "[Connect] bad response: " << status << endl;
            result = false;
        } else {
            logger.post(DIAG_INFORMATION) << "[Connect] response OK" << endl;
            processQueryResponse(this->pConnection, qConnect, data);

            if (messageId.empty() || !initialized || message.size()) {
                logger.post(DIAG_ERROR) << "[Connect] response data is invalid " << endl;
                result = false;
            } else {
                logger.post(DIAG_DEBUG) << "messageId: " << messageId << endl;
                result = true;
                asyncReceive.start();
            }
        }
    } catch (TimeoutException& timeoutEx) {
        logger.post(DIAG_ERROR) << "[Connect] error: " << timeoutEx.displayText() << endl;
        result = false;
    } catch (Poco::Exception& e) {
        logger.post(DIAG_ERROR) << "[Connect] error: " << e.displayText() << endl;
        result = false;
    }
    connectSession.getSession()->abort();
    logger.post(DIAG_DEBUG) << "exiting -- " << __func__ << endl;
    return result;
}

void LongPollingTransport::doDisconnect() {
    logger.post(DIAG_DEBUG) << "entering -- " << __func__ << endl;
    if (asyncReceive.isRunning()) {
        asyncReceive.stop();
        this->pConnection->doAbort();
        asyncReceive.wait();
    }
    logger.post(DIAG_DEBUG) << "exiting -- " << __func__ << endl;
}

void LongPollingTransport::doSend(const string& message) {
    bool result=false;
    string encodedMsg;
    ClientSession sendSession(pConnection->host, pConnection->port, pConnection->secure);
    sendSession.getSession()->setKeepAlive(true);

    logger.post(DIAG_DEBUG) << "entering -- " << __func__ << endl;
    try {
        HTTPRequest request(HTTPRequest::HTTP_POST, createQuery(this->pConnection, qSend), HTTPMessage::HTTP_1_1);

        HTMLForm form;
        form.add("data", message);
        form.prepareSubmit(request);

        ostream& transmitChannel = sendSession.getSession()->sendRequest(request);
        form.write(transmitChannel);

        //request.write(cout);

        HTTPResponse response;
        istream& data = sendSession.getSession()->receiveResponse(response);
        HTTPResponse::HTTPStatus status = response.getStatus();
        if (status != HTTPResponse::HTTP_OK) {
            //error: bad response
            logger.post(DIAG_ERROR) << "[Send] bad response: " << status << endl;
            result = false;
        } else {
            logger.post(DIAG_INFORMATION) << "[Send] response OK" << endl;
            result = true;
        }
    } catch (TimeoutException& timeoutEx) {
        logger.post(DIAG_ERROR) << "[Send] error: " << timeoutEx.displayText() << endl;
        result = false;
    } catch (Poco::Exception& e) {
        logger.post(DIAG_ERROR) << "[Send] error: " << e.displayText() << endl;
        result = false;
    }
    sendSession.getSession()->abort();

    if (!result)
        pConnection->notificationCenter.postNotification(new ErrorEvent("Unable to send message"));
    
    logger.post(DIAG_DEBUG) << "exiting -- " << __func__ << endl;
}

void LongPollingTransport::doReceive() {
    bool disconnected=false;
    ClientSession receiveSession(pConnection->host, pConnection->port, pConnection->secure);
    Timespan pollTimeout((pConnection->pollTimeout)*1000*1000);
    receiveSession.getSession()->setTimeout(pollTimeout);

    logger.post(DIAG_DEBUG) << "entering -- " << __func__ << endl;
    logger.post(DIAG_INFORMATION) << "poll timeout " << pConnection->pollTimeout << endl;

    while(!asyncReceive.isStopped() && disconnected == false) {

        try {
            HTTPRequest request(HTTPRequest::HTTP_GET, createQuery(this->pConnection, qPoll), HTTPMessage::HTTP_1_1);
            receiveSession.getSession()->sendRequest(request);

            //request.write(cout);

            HTTPResponse response;
            istream& data = receiveSession.getSession()->receiveResponse(response);
            HTTPResponse::HTTPStatus status = response.getStatus();
            if (status != HTTPResponse::HTTP_OK) {
                //error: bad response
                logger.post(DIAG_ERROR) << "[Poll] bad response: " << status << endl;
                // trigger reconnect
                if (doReconnect() == true) {
                    disconnected=false;
                } else {
                    disconnected=true;
                }
            } else {
                logger.post(DIAG_INFORMATION) << "[Poll] response OK" << endl;
                processQueryResponse(this->pConnection, qPoll, data);

                logger.post(DIAG_DEBUG) << "message array size: " << message.size() << endl;
                for(int i = 0; i < message.size(); i++) {
                    if (message.isArray(i)) {
                        logger.post(DIAG_WARNING) << "!!! Array of json objects in message is not supported yet !!!" << endl;
                    } else if (message.isObject(i)) {
                        logger.post(DIAG_INFORMATION) << "json object in message" << endl;
                        stringstream strObject;
                        Object::Ptr jsonObject = message.getObject(i);
                        jsonObject->stringify(strObject);
                        pConnection->notificationCenter.postNotification(new MessageReceivedEvent(strObject.str()));
                    } else {
                        string strMessage = message.getElement<string>(i);
                        logger.post(DIAG_DEBUG) << "Element: " << strMessage << endl;
                        logger.post(DIAG_DEBUG) << "observers: " << pConnection->notificationCenter.countObservers() << endl;
                        pConnection->notificationCenter.postNotification(new MessageReceivedEvent(strMessage));
                    }
                }
                if (pConnection->longPollDelay > 0)
                    Thread::sleep(pConnection->longPollDelay*1000);
            }
        } catch (TimeoutException& timeoutEx) {
            logger.post(DIAG_WARNING) << "[Poll] error: " << timeoutEx.displayText() << endl;
            // trigger reconnect
            if (!asyncReceive.isStopped() && doReconnect() == true) {
                disconnected=false;
            } else {
                disconnected=true;
            }
        } catch (Poco::Exception& e) {
            logger.post(DIAG_WARNING) << "[Poll] error: " << e.displayText() << endl;
            // trigger reconnect
            if (!asyncReceive.isStopped() && doReconnect() == true) {
                disconnected=false;
            } else {
                disconnected=true;
            }
        }
    }

    if (disconnected)
        pConnection->notificationCenter.postNotification(new ErrorEvent("Unable to receive"));
    receiveSession.getSession()->abort();
    logger.post(DIAG_DEBUG) << "exiting -- " << __func__ << endl;
}

bool LongPollingTransport::doReconnect() {

    if (!(pConnection->syncDisconnectLock.tryReadLock())) {
        logger.post(DIAG_DEBUG) << "Unable to reconnect: Could not acquire lock; disconnection in progress" << endl;
        return false;
    }
    pConnection->syncLock.lock();

    bool result = false;
    Stopwatch disconnectStopwatch;
    long timeDiff=0;
    ClientSession reconnectSession(pConnection->host, pConnection->port, pConnection->secure);
    reconnectSession.getSession()->setTimeout(Timespan(pConnection->disconnectTimeout*1000*1000));
    reconnectSession.getSession()->setKeepAlive(true);

    logger.post(DIAG_DEBUG) << "entering -- " << __func__ << endl;
    logger.post(DIAG_INFORMATION) << "disconnect timeout " << pConnection->disconnectTimeout << endl;

    pConnection->setState(sReconnecting);
    pConnection->notificationCenter.postNotification(new ReconnectingEvent());

    disconnectStopwatch.start();

    do {
        try {
            HTTPRequest request(HTTPRequest::HTTP_GET, createQuery(this->pConnection, qReconnect), HTTPMessage::HTTP_1_1);
            reconnectSession.getSession()->sendRequest(request);

            //request.write(cout);

            HTTPResponse response;
            istream& data = reconnectSession.getSession()->receiveResponse(response);
            HTTPResponse::HTTPStatus status = response.getStatus();
            if (status != HTTPResponse::HTTP_OK) {
                //error: bad response
                logger.post(DIAG_ERROR) << "[Reconnect] bad response: " << status << endl;
                result = false;
            } else {
                logger.post(DIAG_INFORMATION) << "[Reconnect] response OK" << endl;
                result = true;
                processQueryResponse(this->pConnection, qReconnect, data);
                logger.post(DIAG_DEBUG) << "message array size: " << message.size() << endl;
                for(int i = 0; i < message.size(); i++) {
                    if (message.isArray(i)) {
                        logger.post(DIAG_WARNING) << "!!! Array of json objects in message !!!" << endl;
                    } else if (message.isObject(i)) {
                        logger.post(DIAG_INFORMATION) << "json object in message" << endl;
                        stringstream strObject;
                        Object::Ptr jsonObject = message.getObject(i);
                        jsonObject->stringify(strObject);
                        pConnection->notificationCenter.postNotification(new MessageReceivedEvent(strObject.str()));
                    } else {
                        string strMessage = message.getElement<string>(i);
                        logger.post(DIAG_DEBUG) << "Element: " << strMessage << endl;
                        logger.post(DIAG_DEBUG) << "observers: " << pConnection->notificationCenter.countObservers() << endl;
                        pConnection->notificationCenter.postNotification(new MessageReceivedEvent(strMessage));
                    }
                }
            }
        } catch (TimeoutException& timeoutEx) {
            logger.post(DIAG_ERROR) << "[Reconnect] error: " << timeoutEx.displayText() << endl;
            result = false;
        } catch (Poco::Exception& e) {
            logger.post(DIAG_ERROR) << "[Reconnect] error: " << e.displayText() << endl;
            result = false;
        }
        timeDiff = pConnection->disconnectTimeout - disconnectStopwatch.elapsedSeconds();
        if (timeDiff > 0) {
            reconnectSession.getSession()->setTimeout(Timespan(timeDiff*1000*1000));
        }
    } while(timeDiff > 0);

    reconnectSession.getSession()->abort();

    if (result == true) {
        pConnection->setState(sConnected);
        pConnection->notificationCenter.postNotification(new ReconnectedEvent());
    } else {
        pConnection->setState(sDisconnected);
        pConnection->notificationCenter.postNotification(new DisconnectedEvent());
        pConnection->notificationCenter.postNotification(new ErrorEvent("Unable to reconnect"));
    }
    logger.post(DIAG_DEBUG) << "exiting -- " << __func__ << endl;
    pConnection->syncLock.unlock();
    pConnection->syncDisconnectLock.unlock();
    return result;
}

}
