/*

Copyright 2015, Mentor Graphics Corporation
http://www.mentor.com

*/

#include <iostream>
#include "Poco/NObserver.h"
#include "signalRClient/ConnectionLife.h"
#include "signalRClient/Connection.h"

using std::endl;
using Poco::TimerCallback;
using Poco::NObserver;

namespace signalRClient {

ConnectionLife::ConnectionLife(Connection* pConnection) 
: pConnection(pConnection) {
    warnTimeout = this->pConnection->getKeepAliveTimeout()*CONNECTIONSLOW_WARN_TIMEOUT;
    checkTimeout = (this->pConnection->getKeepAliveTimeout()-warnTimeout)*KEEPALIVE_CHECK_TIMEOUT;
    warnTimeout *= TO_MICROSECOND;
    checkTimeout *= TO_MILLISECOND;
    slowReported = false;

    pConnection->notificationCenter.addObserver(NObserver<ConnectionLife, KeepAliveEvent>(*this, &ConnectionLife::onKeepAlive));
}

ConnectionLife::~ConnectionLife() {
    timer.stop();
    pConnection->notificationCenter.removeObserver(NObserver<ConnectionLife, KeepAliveEvent>(*this, &ConnectionLife::onKeepAlive));
    pConnection->diagnosticLogger.post(DIAG_DEBUG) << "Stopping keep-alive monitoring" << endl;
}

void ConnectionLife::start() {
    pConnection->diagnosticLogger.post(DIAG_DEBUG) << "Starting to listen for keep-alive messages" << endl;
    TimerCallback<ConnectionLife> callback(*(this), &ConnectionLife::onCheckLife);
    timer.setPeriodicInterval(checkTimeout);
    timer.start(callback);
}

void ConnectionLife::onCheckLife(Timer&) {
    Timestamp currentTime;
    Timestamp::TimeDiff elapsedmS = (currentTime - lastKeepAlive);
    long keepAliveTimeoutmS = this->pConnection->getKeepAliveTimeout()*TO_MICROSECOND;
    if (elapsedmS > keepAliveTimeoutmS) {
        //issue transport re-connect
        slowReported = false;
        pConnection->pTransport->asyncReconnect(0);
    } else if ((elapsedmS > warnTimeout) && !slowReported) {
        //issue connection slow
        slowReported = true;
        pConnection->notificationCenter.postNotification(new ConnectionSlowEvent());
    }
}

void ConnectionLife::onKeepAlive(const AutoPtr<KeepAliveEvent>& event) {
    pConnection->diagnosticLogger.post(DIAG_DEBUG) << "Got keep-alive message" << endl;
    lastKeepAlive.update();
}

}
