/*

Copyright 2015, Mentor Graphics Corporation
http://www.mentor.com


*/

#include "signalRClient/ConnectionEvents.h"

namespace signalRClient {

ConnectionEventsListener::ConnectionEventsListener() 
: dispatcher(this, &ConnectionEventsListener::dispatchEvents) {

    dispatcher.start();
}

ConnectionEventsListener::~ConnectionEventsListener() {

    //must unsubscribe the subscribed events before deleting listener
    while (!listenerQueue.empty())
        Poco::Thread::sleep(100);
    
    if (dispatcher.isRunning()) {
        //wakeup after stop
        dispatcher.stop();
        listenerQueue.wakeUpAll();
        dispatcher.wait();
    }
}

void ConnectionEventsListener::dispatchEvents() {

    while (!dispatcher.isStopped()) {

        AutoPtr<Notification> pEvent(listenerQueue.waitDequeueNotification());
        while (pEvent) {

            AutoPtr<MessageReceivedEvent> pMessageReceivedEvent = pEvent.cast<MessageReceivedEvent>();
            if (!pMessageReceivedEvent.isNull()) {
                this->onMessageReceived(pMessageReceivedEvent);
                break;
            }

            AutoPtr<ConnectedEvent> pConnectedEvent = pEvent.cast<ConnectedEvent>();
            if (!pConnectedEvent.isNull()) {
                this->onConnected(pConnectedEvent);
                break;
            }

            AutoPtr<ConnectionSlowEvent> pConnectionSlowEvent = pEvent.cast<ConnectionSlowEvent>();
            if (!pConnectionSlowEvent.isNull()) {
                this->onConnectionSlow(pConnectionSlowEvent);
                break;
            }

            AutoPtr<ReconnectingEvent> pReconnectingEvent = pEvent.cast<ReconnectingEvent>();
            if (!pReconnectingEvent.isNull()) {
                this->onReconnecting(pReconnectingEvent);
                break;
            }

            AutoPtr<ReconnectedEvent> pReconnectedEvent = pEvent.cast<ReconnectedEvent>();
            if (!pReconnectedEvent.isNull()) {
                this->onReconnected(pReconnectedEvent);
                break;
            }

            AutoPtr<DisconnectedEvent> pDisconnectedEvent = pEvent.cast<DisconnectedEvent>();
            if (!pDisconnectedEvent.isNull()) {
                this->onDisconnected(pDisconnectedEvent);
                break;
            }

            AutoPtr<ErrorEvent> pErrorEvent = pEvent.cast<ErrorEvent>();
            if (!pErrorEvent.isNull()) {
                this->onError(pErrorEvent);
            }

            pEvent = listenerQueue.waitDequeueNotification();
        }
    }
}

void ConnectionEventsListener::notify(const AutoPtr<Notification>& event) {
    listenerQueue.enqueueNotification(event);
}

void ConnectionEventsListener::notifyOnMessageReceived(const AutoPtr<MessageReceivedEvent>& event) {
    notify(event);
}

void ConnectionEventsListener::notifyOnConnected(const AutoPtr<ConnectedEvent>& event) {
    notify(event);
}

void ConnectionEventsListener::notifyOnConnectionSlow(const AutoPtr<ConnectionSlowEvent>& event) {
    notify(event);
}

void ConnectionEventsListener::notifyOnReconnecting(const AutoPtr<ReconnectingEvent>& event) {
    notify(event);
}

void ConnectionEventsListener::notifyOnReconnected(const AutoPtr<ReconnectedEvent>& event) {
    notify(event);
}

void ConnectionEventsListener::notifyOnDisconnected(const AutoPtr<DisconnectedEvent>& event) {
    notify(event);
}

void ConnectionEventsListener::notifyOnError(const AutoPtr<ErrorEvent>& event) {
    notify(event);
}

}
