/*

Copyright 2015, Mentor Graphics Corporation
http://www.mentor.com

*/

#include <iostream>
#include "Poco/NObserver.h"
#include "Poco/AutoPtr.h"
#include "Poco/URI.h"
#include "Poco/Net/HTTPRequest.h"
#include "Poco/Net/HTTPResponse.h"
#include "Poco/Timespan.h"
#include "signalRClient/Connection.h"
#include "signalRClient/Transport.h"
#include "signalRClient/WebSocketsTransport.h"
#include "signalRClient/LongPollingTransport.h"
#include "signalRClient/SignalRException.h"
#include "signalRClient/ClientSession.h"
#include "signalRClient/ConnectionLife.h"

using std::endl;
using std::stringstream;
using Poco::NObserver;
using Poco::AutoPtr;
using Poco::URI;
using Poco::Net::HTTPRequest;
using Poco::Net::HTTPResponse;
using Poco::Net::HTTPMessage;
using Poco::Timespan;

namespace signalRClient {

Connection::Connection()
: currentState(sDisconnected)
, baseUrl(BASE_URL)
, connectionTimeout(CONNECTION_TIMEOUT)
, keepAliveTimeout(KEEPALIVE_TIMEOUT)
, disconnectTimeout(DISCONNECT_TIMEOUT)
, tryWebSockets(TRY_WEBSOCKETS)
, protocolVersion(SIGNALR_CLIENT_PROTOCOL_VERSION)
, transportConnectTimeout(TRANSPORTCONNECT_TIMEOUT)
, longPollDelay(LONGPOLL_DELAY)
, pollTimeout(POLL_TIMEOUT)
, transportType(TRANSPORT_TYPE)
, pTransport(0)
, asyncStart(this, &Connection::doStart)
, asyncStop(this, &Connection::doStop)
, asyncSend(this, &Connection::doSend)
, scheme(BASE_SCHEME)
, port(BASE_PORT)
, secure(false) 
, diagnosticLogger(SIGNALR_CLIENT_LOGGER)
, stopped(false) {

}

Connection::Connection(const string& uri)
: uri(uri)
, currentState(sDisconnected)
, baseUrl(BASE_URL)
, connectionTimeout(CONNECTION_TIMEOUT)
, keepAliveTimeout(KEEPALIVE_TIMEOUT)
, disconnectTimeout(DISCONNECT_TIMEOUT)
, tryWebSockets(TRY_WEBSOCKETS)
, protocolVersion(SIGNALR_CLIENT_PROTOCOL_VERSION)
, transportConnectTimeout(TRANSPORTCONNECT_TIMEOUT)
, longPollDelay(LONGPOLL_DELAY)
, pollTimeout(POLL_TIMEOUT)
, transportType(TRANSPORT_TYPE)
, pTransport(0)
, asyncStart(this, &Connection::doStart)
, asyncStop(this, &Connection::doStop)
, asyncSend(this, &Connection::doSend)
, scheme(BASE_SCHEME)
, port(BASE_PORT) 
, secure(false) 
, diagnosticLogger(SIGNALR_CLIENT_LOGGER)
, stopped(false) {

    parseURI();
}

Connection::Connection(const string& uri, const string& query)
: uri(uri)
, currentState(sDisconnected)
, baseUrl(BASE_URL)
, connectionTimeout(CONNECTION_TIMEOUT)
, keepAliveTimeout(KEEPALIVE_TIMEOUT)
, disconnectTimeout(DISCONNECT_TIMEOUT)
, tryWebSockets(TRY_WEBSOCKETS)
, protocolVersion(SIGNALR_CLIENT_PROTOCOL_VERSION)
, transportConnectTimeout(TRANSPORTCONNECT_TIMEOUT)
, longPollDelay(LONGPOLL_DELAY)
, pollTimeout(POLL_TIMEOUT)
, transportType(TRANSPORT_TYPE)
, pTransport(0)
, asyncStart(this, &Connection::doStart)
, asyncStop(this, &Connection::doStop)
, asyncSend(this, &Connection::doSend)
, scheme(BASE_SCHEME)
, port(BASE_PORT) 
, secure(false) 
, customQuery(query) 
, diagnosticLogger(SIGNALR_CLIENT_LOGGER)
, stopped(false) {

    parseURI();
}

Connection::~Connection() {

    if (!stopped) {
        stop().wait();
    }

    if(pTransport) delete pTransport;
    pTransport = 0;
}

void Connection::setState(const ConnectionState& newState) {
    this->currentState = newState;
}

const ConnectionState& Connection::getState() const {
    return this->currentState;
}

const Transport* Connection::getTransport() const {
    return pTransport;
}


ActiveResult<void> Connection::send(const string& message) {
    return asyncSend(message);
}

ActiveResult<void> Connection::sendObject(const Object& message) {

    stringstream outStr;
    try {
        message.stringify(outStr);
        return asyncSend(outStr.str());
    } catch (Poco::Exception& pocoEx) {
        diagnosticLogger.post(DIAG_ERROR) << "Unable to send message: " << pocoEx.displayText() << endl;
        throw SignalRException("Unable to send message");
    }
}

void Connection::doSend(const string& message) {
    
    if(!syncDisconnectLock.tryReadLock()) {
        diagnosticLogger.post(DIAG_DEBUG) << "Unable to send message: could not acquire lock" << endl;
        return;
    }

    syncLock.lock();

    if((this->currentState == sConnected) && pTransport && !message.empty()) {
        pTransport->doSend(message);
    } else {
        diagnosticLogger.post(DIAG_ERROR) << "Unable to send message: " << (message.empty() ? "Message is empty" : "Transport connection is not initialized yet") << endl;
        notificationCenter.postNotification(new ErrorEvent("Unable to send message"));
    }

    syncLock.unlock();
    syncDisconnectLock.unlock();
}

void Connection::subscribeListenertoEvent(ConnectionEventsListener* listener, ConnectionEventType eventType) {

    if (!listener) {
        throw SignalRException("<listener> is NULL");
    }

    try {
        switch (eventType) {

            case eMessageReceived:
                notificationCenter.addObserver(NObserver<ConnectionEventsListener, MessageReceivedEvent>(*listener, &ConnectionEventsListener::notifyOnMessageReceived));
            break;
            case eConnected:
                notificationCenter.addObserver(NObserver<ConnectionEventsListener, ConnectedEvent>(*listener, &ConnectionEventsListener::notifyOnConnected));
            break;
            case eConnectionSlow:
                notificationCenter.addObserver(NObserver<ConnectionEventsListener, ConnectionSlowEvent>(*listener, &ConnectionEventsListener::notifyOnConnectionSlow));
            break;
            case eReconnecting:
                notificationCenter.addObserver(NObserver<ConnectionEventsListener, ReconnectingEvent>(*listener, &ConnectionEventsListener::notifyOnReconnecting));
            break;
            case eReconnected:
                notificationCenter.addObserver(NObserver<ConnectionEventsListener, ReconnectedEvent>(*listener, &ConnectionEventsListener::notifyOnReconnected));
            break;
            case eDisconnected:
                notificationCenter.addObserver(NObserver<ConnectionEventsListener, DisconnectedEvent>(*listener, &ConnectionEventsListener::notifyOnDisconnected));
            break;
            case eError:
                notificationCenter.addObserver(NObserver<ConnectionEventsListener, ErrorEvent>(*listener, &ConnectionEventsListener::notifyOnError));
            break;
            default:
                throw SignalRException("<eventType> is unknown");
        }
    } catch (Poco::Exception& pocoEx) {
        throw SignalRException(pocoEx.displayText());
    }
}

void Connection::unSubscribeListenerfromConnectionEvent(ConnectionEventsListener* listener, ConnectionEventType eventType) {

    if (!listener) {
        throw SignalRException("<listener> is NULL");
    }

    try {
        switch (eventType) {

            case eMessageReceived:
                notificationCenter.removeObserver(NObserver<ConnectionEventsListener, MessageReceivedEvent>(*listener, &ConnectionEventsListener::notifyOnMessageReceived));
            break;
            case eConnected:
                notificationCenter.removeObserver(NObserver<ConnectionEventsListener, ConnectedEvent>(*listener, &ConnectionEventsListener::notifyOnConnected));
            break;
            case eConnectionSlow:
                notificationCenter.removeObserver(NObserver<ConnectionEventsListener, ConnectionSlowEvent>(*listener, &ConnectionEventsListener::notifyOnConnectionSlow));
            break;
            case eReconnecting:
                notificationCenter.removeObserver(NObserver<ConnectionEventsListener, ReconnectingEvent>(*listener, &ConnectionEventsListener::notifyOnReconnecting));
            break;
            case eReconnected:
                notificationCenter.removeObserver(NObserver<ConnectionEventsListener, ReconnectedEvent>(*listener, &ConnectionEventsListener::notifyOnReconnected));
            break;
            case eDisconnected:
                notificationCenter.removeObserver(NObserver<ConnectionEventsListener, DisconnectedEvent>(*listener, &ConnectionEventsListener::notifyOnDisconnected));
            break;
            case eError:
                notificationCenter.removeObserver(NObserver<ConnectionEventsListener, ErrorEvent>(*listener, &ConnectionEventsListener::notifyOnError));
            break;
            default:
                throw SignalRException("<eventType> is unknown");
        }
    } catch (Poco::Exception& pocoEx) {
        throw SignalRException(pocoEx.displayText());
    }
}

void Connection::doStart(const TransportType& transportType) {

    if(!syncDisconnectLock.tryReadLock()) {
        diagnosticLogger.post(DIAG_DEBUG) << "Unable to start connection: Could not acquire lock; disconnection in progress" << endl;
        return;
    }

    syncLock.lock();

    try {
        if (pTransport) delete pTransport;
        pTransport = 0;

        setState(sConnecting);
        doNegotiation();
        if (stopped) {
            throw SignalRException("Connection stopped by user");
        }

        doConnection(transportType);
        notificationCenter.postNotification(new ConnectedEvent());
        setState(sConnected);
    } catch (SignalRException& error) {
        setState(sDisconnected);
        diagnosticLogger.post(DIAG_FATAL) << "start terminated with error: " << error.getMessage() << endl;
        notificationCenter.postNotification(new ErrorEvent("Unable to start connection"));
    }

    syncLock.unlock();
    syncDisconnectLock.unlock();
}

void Connection::doNegotiation() {

    try {
        ClientSession cs(host, port, secure);
        const string& negotiateQ = createQuery(this, qNegotiate);
        diagnosticLogger.post(DIAG_DEBUG) << "Issuing query: " << negotiateQ << endl;
        HTTPRequest request(HTTPRequest::HTTP_GET, negotiateQ, HTTPMessage::HTTP_1_1);
        cs.getSession()->sendRequest(request);

        HTTPResponse response;
        istream& data = cs.getSession()->receiveResponse(response);
        HTTPResponse::HTTPStatus status = response.getStatus();
        if (status != HTTPResponse::HTTP_OK) {
             diagnosticLogger.post(DIAG_ERROR) << "Unable to /negotiate: bad response " << response.getReason() << endl;
            throw SignalRException("Unable to /negotiate");
        }
        processQueryResponse(this, qNegotiate, data);
    } catch (Poco::Exception& pocoEx) {
        diagnosticLogger.post(DIAG_ERROR) << "Unable to /negotiate: " << pocoEx.displayText() << endl;
        throw SignalRException("Unable to /negotiate");
    }

    diagnosticLogger.post(DIAG_DEBUG) << "/negotiate is successful" << endl;
}

void Connection::doConnection(const TransportType& transportType) {

    if(transportType < tAuto || transportType > tLongPolling) {
        throw SignalRException("<transportType> is unknown");
    }

    if((transportType == tWebSockets) && !tryWebSockets) {
        throw SignalRException("WebSockets are not supported by the SignalR server");
    }

    TransportType negotiatedTransport;

    switch(transportType) {

        case tAuto:
        case tWebSockets:
        {
            diagnosticLogger.post(DIAG_DEBUG) << "Trying to connect with WebSockets transport" << endl;
            if(tryWebSockets) {
                this->pTransport = (Transport*) new WebSocketsTransport(this);

                if(this->pTransport->doConnect()) {
                    if(doStart()) {
                        negotiatedTransport = tWebSockets;
                        break;
                    }
                }
            
                delete this->pTransport;
                pTransport = 0;
            }

            if (stopped) {
                throw SignalRException("Connection stopped by user");
            } else {
                diagnosticLogger.post(DIAG_INFORMATION) << "WebSockets are not supported by the SignalR server, falling back to Long Polling" << endl;
            }
        }
        case tLongPolling:
        {
            diagnosticLogger.post(DIAG_DEBUG) << "Trying to connect with Long Polling transport" << endl;
            this->pTransport = (Transport*) new LongPollingTransport(this);
       
            if(this->pTransport->doConnect()) {
                if(doStart()) {
                    negotiatedTransport = tLongPolling;
                    break;
                }
            }
            
            delete this->pTransport;
            pTransport = 0;

            if (stopped) {
                throw SignalRException("Connection stopped by user");
            } 
        }
        default:
        {
            diagnosticLogger.post(DIAG_FATAL) << "Failed to connect to SignalR server on WebSockets and Long Polling transport" << endl;
            throw SignalRException("Failed to connect to SignalR server on WebSockets and Long Polling transport");
        }
    }

    setTransportType(negotiatedTransport);
    diagnosticLogger.post(DIAG_DEBUG) << this->pTransport->getName() << " transport connected successfully" << endl;
}

void Connection::doStop() {

    stopped = true;
    syncDisconnectLock.writeLock();
    stopped = false;

    if (this->getState() != sDisconnected) {

        try {
            if (pTransport) pTransport->doDisconnect();
            notificationCenter.postNotification(new DisconnectedEvent());
        } catch (SignalRException& error) {
            diagnosticLogger.post(DIAG_ERROR) << "stop terminated with error: " << error.getMessage() << endl;
            notificationCenter.postNotification(new ErrorEvent("Unable to stop connection"));
        }
        
        if (pTransport) delete pTransport;
        pTransport = 0;
        setState(sDisconnected);
    }

    syncDisconnectLock.unlock();
}

void Connection::setQueryString(const string& query) {
    this->customQuery = query;
}

const string& Connection::getQueryString() {
    return this->customQuery;
}

void Connection::parseURI() {

    if(this->uri.empty()) {
        throw SignalRException("<uri> is empty");
    }

    try {
        URI pocoUri(this->uri);

        host = pocoUri.getHost();
        port = pocoUri.getPort();
        baseUrl = pocoUri.getPath().empty() ? baseUrl : pocoUri.getPath();

        if(host.empty()) {
            throw SignalRException("Malformed <uri>");
        }

        if(pocoUri.getScheme() == "https") {
            scheme = pocoUri.getScheme();
            secure = true;
        }
    } catch (Poco::Exception& pocoEx) {
        throw SignalRException(pocoEx.displayText());
    }
}

bool Connection::doStart() {

    if (stopped) {
        throw SignalRException("Connection stopped by user");
    }

    try {
        ClientSession cs(host, port, secure);
        const string& startQ = createQuery(this, qStart);
        diagnosticLogger.post(DIAG_DEBUG) << "Issuing query: " << startQ << endl;
        HTTPRequest startRequest(HTTPRequest::HTTP_GET, startQ, HTTPMessage::HTTP_1_1);
        cs.getSession()->sendRequest(startRequest);

        HTTPResponse startResp;
        istream& data = cs.getSession()->receiveResponse(startResp);
        HTTPResponse::HTTPStatus status = startResp.getStatus();
        if (status != HTTPResponse::HTTP_OK) {
            diagnosticLogger.post(DIAG_ERROR) << "Unable to /start: bad response " << startResp.getReason() << endl;
            throw SignalRException("Unable to /start");
        }

        processQueryResponse(this, qStart, data);
        if (startResponse != "started") {
            diagnosticLogger.post(DIAG_ERROR) << "Unable to /start: invalid response " << startResponse << endl;
            throw SignalRException("Unable to /start");
        }
    } catch (Poco::Exception& pocoEx) {
        diagnosticLogger.post(DIAG_ERROR) << "Unable to /start: " << pocoEx.displayText() << endl;
        throw SignalRException("Unable to /start");
    }

    diagnosticLogger.post(DIAG_DEBUG) << "/start is successful" << endl;
    return true;
}

void Connection::doAbort() {

    try {
        ClientSession cs(host, port, secure);
        const string& abortQ = createQuery(this, qAbort);
        diagnosticLogger.post(DIAG_DEBUG) << "Issuing query: " << abortQ << endl;
        HTTPRequest request(HTTPRequest::HTTP_GET, abortQ, HTTPMessage::HTTP_1_1);
        cs.getSession()->sendRequest(request);

        HTTPResponse response;
        istream& data = cs.getSession()->receiveResponse(response);
        HTTPResponse::HTTPStatus status = response.getStatus();
        if (status != HTTPResponse::HTTP_OK) {
            diagnosticLogger.post(DIAG_ERROR) << "Unable to /abort: bad response " << response.getReason() << endl;
            throw SignalRException("Unable to /abort");
        }

        int ch = data.get();
        if (ch == '\r') ch = data.get();
        if (ch == '\n') ch = data.get();
        if (ch != std::char_traits<char>::eof()) {
            diagnosticLogger.post(DIAG_ERROR) << "Unable to /abort: query resulted in non empty body" << endl;
            throw SignalRException("Unable to /abort");
        }
    } catch (Poco::Exception& pocoEx) {
        diagnosticLogger.post(DIAG_ERROR) << "Unable to /abort: " << pocoEx.displayText() << endl;
        throw SignalRException("Unable to /abort");
    }
    diagnosticLogger.post(DIAG_DEBUG) << "/abort is successful" << endl;
}

void Connection::setLogLevel(const DIAG_LEVEL& level) {
    try {
        diagnosticLogger.setLevel(level);
    } catch (Poco::Exception& pocoEx) {
        throw SignalRException(pocoEx.displayText());
    }
}

const DIAG_LEVEL& Connection::getLogLevel() {
    return diagnosticLogger.getLevel();
}

void Connection::setLogFile(const string& filename) {

    try {
        diagnosticLogger.setFile(filename);
    } catch (Poco::Exception& pocoEx) {
        throw SignalRException(pocoEx.displayText());
    }
}

}
