/*

Copyright 2015, Mentor Graphics Corporation
http://www.mentor.com


*/

#include "signalRClient/Connection.h"
#include "signalRClient/ConnectionEvents.h"
#include "signalRClient/ConnectionState.h"
#include "signalRClient/Transport.h"
#include "signalRClient/SignalRException.h"

#include "Poco/Util/Option.h"
#include "Poco/Util/OptionSet.h"
#include "Poco/Util/OptionProcessor.h"
#include "Poco/Util/OptionException.h"
#include "Poco/FileStream.h"

#include <unistd.h>

using namespace std;
using namespace signalRClient;

using Poco::Util::Option;
using Poco::Util::OptionSet;
using Poco::Util::OptionProcessor;
using Poco::FileInputStream;

class CustomConnectionEventsListener: public ConnectionEventsListener
{
    int messageReceiveCounter = 0;
    int connectCounter = 0;
    int connectionSlowCounter = 0;
    int reconnectingCounter = 0;
    int reconnectedCounter = 0;
    int disconnectCounter = 0;
    int erroCounter = 0;

public:
    void onMessageReceived(const AutoPtr<MessageReceivedEvent>& event) {
        cout << "[Event] SignalR client has received a message: " << event->getMessage() << endl;
        messageReceiveCounter++;
    }

    void onConnected(const AutoPtr<ConnectedEvent>& event) {
        cout << "[Event] SignalR client is now connected to the SignalR server!" << endl;
        connectCounter++;
    }

    void onConnectionSlow(const AutoPtr<ConnectionSlowEvent>& event) {
        cout << "[Event] Network connection between SignalR client and server seems to be slow" << endl;
        connectionSlowCounter++;
    }

    void onReconnecting(const AutoPtr<ReconnectingEvent>& event) {
        cout << "[Event] SignalR client is reconnecting to the SignalR server!" << endl;
        reconnectingCounter++;
    }

    void onReconnected(const AutoPtr<ReconnectedEvent>& event) {
        cout << "[Event] Connection between SignalR client and server has been re-established!" << endl;
        reconnectedCounter++;
    }

    void onDisconnected(const AutoPtr<DisconnectedEvent>& event) {
        cout << "[Event] SignalR client has been disconnected from the SignalR server!" << endl;
        disconnectCounter++;
    }

    void onError(const AutoPtr<ErrorEvent>& event) {
        cout << "[Event] An error has occurred: " << event->getMessage() << endl;
        erroCounter++;
    }

    void displayStats() {
        cout << "### messages receive counts : " << messageReceiveCounter << " ###" << endl;
        cout << "### connected counts        : " << connectCounter << " ###" << endl;
        cout << "### connection slow counts  : " << connectionSlowCounter << " ###" << endl;
        cout << "### reconnecting counts     : " << reconnectingCounter << " ###" << endl;
        cout << "### reconnected counts      : " << reconnectedCounter << " ###" << endl;
        cout << "### disconnected counts     : " << disconnectCounter << " ###" << endl;
        cout << "### error counts            : " << erroCounter << " ###" << endl;
    }
};

string getAuthenticationToken(const string& authFilename) {

    string authQuery; 
    FileInputStream authFile(authFilename);
    authFile >> authQuery;
    authFile.close();
    return authQuery;
}

void printUsage(OptionSet& options) {

    cout << "Usage:" << endl;
    for (OptionSet::Iterator it = options.begin(); it != options.end(); it++) {

        const Option& option = *it;
        cout << "-" << option.shortName();
        cout << " ";
        cout << "--" << option.fullName();
        cout << " ";
        cout << "[" << option.description() << "]";
        cout << endl;
    }
    cout << endl;
}

void addCLIOptions(OptionSet& optSet) {

    // command line arguments.
    // -h or --help get help on options
    // -u<value> or --uri= for signalr server uri.
    // -t<value> or --transport= transport to use. LONG_POLLING, WEB_SOCKETS
    // -a<value> or --auth= authentication token query.
    // -c<value> or --count= number of random messages to send.
    // -d<value> or --delay= delay between two random messages.
    // -r<value> or --repeat= number of times to repeat the connect.

    optSet.addOption(
            Option("uri", "u", "specify the uri of signalr server")
                .required(false)
                .repeatable(false)
                .argument("uri"));

    optSet.addOption(
            Option("transport", "t", "specify the transport")
                .required(false)
                .repeatable(false)
                .argument("transport"));

    optSet.addOption(
            Option("auth", "a", "specify the file name containing authentication query")
                .required(false)
                .repeatable(false)
                .argument("auth"));

    optSet.addOption(
            Option("count", "c", "specify the number of random messages to send")
                .required(false)
                .repeatable(false)
                .argument("count"));

    optSet.addOption(
            Option("delay", "d", "specify the delay between two random messages")
                .required(false)
                .repeatable(false)
                .argument("delay"));

    optSet.addOption(
            Option("repeat", "r", "specify the number of time to repeat the connect")
                .required(false)
                .repeatable(false)
                .argument("repeat"));

    optSet.addOption(
            Option("help", "h", "get help on available options")
                .required(false)
                .repeatable(false));
}

void processCLIOptions(OptionSet& optSet, int& argc, char** argv, bool& help, string& uri, TransportType& transport,
        string& authQueryFilename, int& messageCount, int& messageDelay, int& repeatCount) {
    OptionProcessor argProcessor(optSet);
    string argName;
    string argValue;

    argProcessor.setUnixStyle(true);

    for (int i=1; i<argc; i++) {
        try {
            argProcessor.process(argv[i], argName, argValue);
            //cout << "argument name: " << argName << endl;
            //cout << "argument value: " << argValue << endl;
            if (argName == "help") {
                help = true;
            }

            if (argName == "uri") {
                if (!argValue.empty()) {
                    cout << "Setting signalR server URI to: " << argValue << endl;
                    uri = argValue;
                }
            }

            if (argName == "transport") {
                if (!argValue.empty()) {
                    if (argValue == "LONG_POLLING") {
                        cout << "Setting transport to: " << argValue << endl;
                        transport = tLongPolling;
                    } else if (argValue == "WEB_SOCKETS") {
                        cout << "Setting transport to: " << argValue << endl;
                        transport = tWebSockets;
                    }
                }
            }

            if (argName == "auth") {
                if (!argValue.empty()) {
                    authQueryFilename = argValue;
                }
            }

            if (argName == "count") {
                if (!argValue.empty()) {
                    messageCount = stoi(argValue);
                }
            }

            if (argName == "delay") {
                if (!argValue.empty()) {
                    messageDelay = stoi(argValue);
                }
            }

            if (argName == "repeat") {
                if (!argValue.empty()) {
                    repeatCount = stoi(argValue);
                }
            }
         } catch (exception e) {
            cout << "argument parsing exception" << endl;
        }
    }
}

void populateMessages(std::vector<std::string>& vec) {
    std::vector<std::string> cols;
    std::vector<std::string> rows;

    cols.push_back("AAAAAAAAAAAAAAAAAAAAAAAAAAAACCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCDDDDDDDDDDDFDFDFDFDFDFDFEERERERERERER"
            "CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCDDDDDDDDDDDFDFDFDFDFDFDFEERERERERERERCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCDDDDDDDDDDDFDFDFDFDFDFDFEERERERERERER"
            "CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCDDDDDDDDDDDFDFDFDFDFDFDFEERERERERERERCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCDDDDDDDDDDDFDFDFDFDFDFDFEERERERERERER"
            "CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCDDDDDDDDDDDFDFDFDFDFDFDFEERERERERERERCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCDDDDDDDDDDDFDFDFDFDFDFDFEERERERERERER"
            "CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCDDDDDDDDDDDFDFDFDFDFDFDFEERERERERERERCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCDDDDDDDDDDDFDFDFDFDFDFDFEERERERERERER"
            "CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCDDDDDDDDDDDFDFDFDFDFDFDFEERERERERERERCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCDDDDDDDDDDDFDFDFDFDFDFDFEERERERERERER"
            "333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333"
            "333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333"
            "333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333"
            "333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333"
            "333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333"
            "333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333330");
    cols.push_back("BBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBB");
    cols.push_back("CCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCCDDDDDDDDDDDFDFDFDFDFDFDFEERERERERERERDDDDDDDDDDDFDFDFDFDFDFDFEERERERERERERDDDDDDDDDDDFDFDFDFDFDFDFEERERERERERER");

    rows.push_back("1111111111111111111111111111111111111111111111111111111111111111111111ssssssssssssssssssssssssssssssssssssssssssssssssssssssssssssssssssssssssssss");
    rows.push_back("22222222222222222222222222222222222222222222222222222222222222222222222222222222");
    rows.push_back("333333333333333333333333333333333333333333333333333333333333333333333333333333333333333333");
    rows.push_back("444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444"
            "444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444"
            "444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444"
            "444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444444"
            "BBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBB"
            "BBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBBB"
            "777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777"
            "777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777"
            "777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777"
            "777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777"
            "777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777"
            "777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777"
            "777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777"
            "777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777"
            "7777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777777555");

    for (auto i : cols)
        for (auto j : rows)
            vec.emplace_back(i + j);
}

int main(int argc, char** argv) {

    try {
        string uri = "http://integrationic3messaging.myicomfort.com/LennoxNotificationServerTest";
        TransportType transport = tAuto;
        string authQueryFilename;
        string authQuery;
        int messageCount = 0;
        int messageDelay = 5;
        int repeatCount = 0;
        int messageSendCounter = 0;

        OptionSet optSet;
        addCLIOptions(optSet);

        bool help = false;

        processCLIOptions(optSet, argc, argv, help, uri, transport, authQueryFilename, messageCount, messageDelay, repeatCount);

        if (help) {
            printUsage(optSet);
            return 0;
        }

        if (!authQueryFilename.empty())
            authQuery = getAuthenticationToken(authQueryFilename);

        Connection connection(uri, authQuery);
        CustomConnectionEventsListener connectionEventsListener;
 
        connection.subscribeListenertoEvent(&connectionEventsListener, eMessageReceived);
        connection.subscribeListenertoEvent(&connectionEventsListener, eConnected);
        connection.subscribeListenertoEvent(&connectionEventsListener, eConnectionSlow);
        connection.subscribeListenertoEvent(&connectionEventsListener, eReconnecting);
        connection.subscribeListenertoEvent(&connectionEventsListener, eReconnected);
        connection.subscribeListenertoEvent(&connectionEventsListener, eDisconnected);
        connection.subscribeListenertoEvent(&connectionEventsListener, eError);

        connection.setLogLevel(DIAG_DEBUG);

        do {
            cout << endl << "### Start Connection with remaining repeat count: " << repeatCount << " ###" << endl << endl;
            connection.start(transport).wait();

            if (connection.getState() == sConnected) {
                cout << "[State] SignalR client has been connected to the SignalR server!" << endl;
                cout << "SignalR client is using transport: " << connection.getTransport()->getName() << endl;

                connection.send("Hello World!");
                messageSendCounter++;

                if (messageCount > 0) {
                    cout << endl << "### Start random message sender. send count: " << messageCount << " ###" << endl << endl;

                    int counter = 0;
                    std::vector<std::string> vec;

                    populateMessages(vec);

                    while(counter < messageCount) {
                        std::random_shuffle(vec.begin(), vec.end());
                        for (auto message : vec) {
                            usleep(messageDelay*1000*1000);
                            connection.send(message);
                            messageSendCounter++;
                            if (++counter >= messageCount)
                                break;
                        }
                    }
                } else {
                    cout << endl << "Start sending messages! Enter <exit> to exit the application" << endl;
                    string input;
                    cin >> input;

                    while (input != "exit") {
                        connection.send(input);
                        messageSendCounter++;
                        cin >> input;
                    }
                }

                connection.stop().wait();
            } else {
                cout << "[State] SignalR client failed to connect to the SignalR server!" << endl;
            }
            cout << endl;
        } while(repeatCount-- > 0);

        cout << endl << endl;
        cout << "### messages send counts    : " << messageSendCounter << " ###" << endl;
        connectionEventsListener.displayStats();
        cout << endl << endl;

    } catch (SignalRException& error) {
        cout << "Something went wrong! error: " << error.getMessage() << endl;
    }

    return 0;
}
