/*

Copyright 2015, Mentor Graphics Corporation
http://www.mentor.com


*/

#ifndef _CONNECTIONEVENTS
#define _CONNECTIONEVENTS

#include <string>
#include "Poco/AutoPtr.h"
#include "Poco/NotificationCenter.h"
#include "Poco/NotificationQueue.h"
#include "Poco/Activity.h"

using std::string;
using Poco::AutoPtr;
using Poco::Notification;
using Poco::NotificationQueue;
using Poco::Activity;

namespace signalRClient {

class Connection;
class MessageReceivedEvent;
class ConnectedEvent;
class ConnectionSlowEvent;
class ReconnectingEvent;
class ReconnectedEvent;
class DisconnectedEvent;
class KeepAliveEvent;
class ErrorEvent;

enum ConnectionEventType {
    eMessageReceived = 0,
    eConnected,
    eConnectionSlow,
    eReconnecting,
    eReconnected,
    eDisconnected,
    eError,
    eLast /* Not available for user*/
};

class ConnectionEventsListener {

public:
    ConnectionEventsListener();
    ~ConnectionEventsListener();

    virtual void onMessageReceived(const AutoPtr<MessageReceivedEvent>& event) { }
    virtual void onConnected(const AutoPtr<ConnectedEvent>& event) { }
    virtual void onConnectionSlow(const AutoPtr<ConnectionSlowEvent>& event) { }
    virtual void onReconnecting(const AutoPtr<ReconnectingEvent>& event) { }
    virtual void onReconnected(const AutoPtr<ReconnectedEvent>& event) { }
    virtual void onDisconnected(const AutoPtr<DisconnectedEvent>& event) { }
    virtual void onError(const AutoPtr<ErrorEvent>& event) { }

private:
    void notify(const AutoPtr<Notification>& event);
    void notifyOnMessageReceived(const AutoPtr<MessageReceivedEvent>& event);
    void notifyOnConnected(const AutoPtr<ConnectedEvent>& event);
    void notifyOnConnectionSlow(const AutoPtr<ConnectionSlowEvent>& event);
    void notifyOnReconnecting(const AutoPtr<ReconnectingEvent>& event);
    void notifyOnReconnected(const AutoPtr<ReconnectedEvent>& event);
    void notifyOnDisconnected(const AutoPtr<DisconnectedEvent>& event);
    void notifyOnError(const AutoPtr<ErrorEvent>& event);

    void dispatchEvents();
    Activity<ConnectionEventsListener> dispatcher;
    NotificationQueue listenerQueue;

    friend class Connection;
};

class MessageReceivedEvent : public Notification {

public:
    MessageReceivedEvent() { }
    MessageReceivedEvent(const string& message) : message(message) { }
    ~MessageReceivedEvent() { }

    void setMessage(const string& message) {
        this->message = message;
    }

    const string& getMessage() const {
        return this->message;
    }

private:
    string message;
};

class ConnectedEvent : public Notification {

};

class ConnectionSlowEvent : public Notification {

};

class ReconnectingEvent : public Notification {

};

class ReconnectedEvent : public Notification {

};

class DisconnectedEvent : public Notification {

};

class ErrorEvent : public Notification {

public:
    ErrorEvent() { }
    ErrorEvent(const string& message) : message(message) { }
    ~ErrorEvent() { }

    void setMessage(const string& message) {
        this->message = message;
    }

    const string& getMessage() const {
        return this->message;
    }

private:
    string message;
};

}

#endif
