/*

Copyright 2015, Mentor Graphics Corporation
http://www.mentor.com

*/

#ifndef _CONNECTION
#define _CONNECTION

#include <string>
#include "Poco/ActiveMethod.h"
#include "Poco/ActiveResult.h"
#include "Poco/JSON/Object.h"
#include "Poco/NotificationCenter.h"
#include "Poco/Mutex.h"
#include "Poco/RWLock.h"
#include "signalRClient/ConnectionState.h"
#include "signalRClient/Transport.h"
#include "signalRClient/ConnectionEvents.h"
#include "signalRClient/Util.h"
#include "signalRClient/Defaults.h"
#include "signalRClient/DiagnosticLogger.h"

using std::string;
using Poco::ActiveMethod;
using Poco::ActiveResult;
using Poco::JSON::Object;
using Poco::NotificationCenter;
using Poco::Mutex;
using Poco::RWLock;

namespace signalRClient {

class ConnectionEventsListener;
class Transport;

class Connection {

public:
    Connection();
    Connection(const string& uri);
    Connection(const string& uri, const string& query);
    ~Connection();

    void setURI(const string& uri) {
        this->uri = uri;
        parseURI();
    }

    const string& getURI() const {
        return this->uri;
    }

    const ConnectionState& getState() const;

    void setTransportType(const TransportType& transportType) {
        this->transportType = transportType;
    }

    const TransportType& getTransportType() const {
        return this->transportType;
    }

    const Transport* getTransport() const;

    ActiveResult<void> start(const TransportType& transportType = tAuto){
        return asyncStart(transportType);
    }

    ActiveResult<void> stop() {
        return asyncStop();
    }

    ActiveResult<void> send(const string& message);
    ActiveResult<void> sendObject(const Object& message);

    void subscribeListenertoEvent(ConnectionEventsListener* listener, ConnectionEventType eventType);
    void unSubscribeListenerfromConnectionEvent(ConnectionEventsListener* listener, ConnectionEventType eventType);

    void setQueryString(const string& query);
    const string& getQueryString();

    const long& getKeepAliveTimeout() const {
        return this->keepAliveTimeout;
    }

    const long& getDisconnectTimeout() const {
        return this->disconnectTimeout;
    }

    const long& getTransportConnectTimeout() const {
        return this->transportConnectTimeout;
    }

    const long& getLongPollDelay() const {
        return this->longPollDelay;
    }

    void setPollTimeout(const long& pollInterval) {
        this->pollTimeout = pollInterval;
    }
    
    const long& getPollTimeout() const {
        return this->pollTimeout;
    }
    const string& getBaseUrl() const {
        return this->baseUrl;
    }

    const string& getConnectionToken() const {
        return this->connectionToken;
    }

    const string& getConnectionId() const {
        return this->connectionId;
    }

    void setConnectionData(const string& connectionData) {
        this->connectionData = connectionData;
    }

    const string& getConnectionData() const {
        return this->connectionData;
    }

    const string& getProtocolVersion() const {
        return this->protocolVersion;
    }

    const bool& getTryWebSockets() const {
        return this->tryWebSockets;
    }

    void setLogLevel(const DIAG_LEVEL& level);
    const DIAG_LEVEL& getLogLevel();

    void setLogFile(const string& filename);

protected:
    DiagnosticLogger diagnosticLogger;
    
private:
    void setState(const ConnectionState& newState);
    
    void doStart(const TransportType& transportType);
    void doNegotiation();
    void doConnection(const TransportType& transportType);
    void doStop();
    void doSend(const string& message);

    string uri;
    ConnectionState currentState;
    string baseUrl;
    string connectionToken;
    string connectionId;
    string connectionData;
    long connectionTimeout;
    long keepAliveTimeout;
    long disconnectTimeout;
    bool tryWebSockets;
    string protocolVersion;
    long transportConnectTimeout;
    long longPollDelay;
    long pollTimeout;
    string customQuery;

    TransportType transportType;
    Transport* pTransport;

    ActiveMethod<void, TransportType, Connection> asyncStart;
    ActiveMethod<void, void, Connection> asyncStop;
    ActiveMethod<void, string, Connection> asyncSend;

    NotificationCenter notificationCenter;

    friend class Transport;
    friend class WebSocketsTransport;
    friend class LongPollingTransport;

    string scheme;
    string host;
    unsigned short port;
    bool secure;
    string startResponse;

    void parseURI();
    bool doStart();

    friend string createQuery(const Connection* pConnection, const QueryType& queryType);
    friend void processQueryResponse(Connection* pConnection, const QueryType& queryType, istream& data);

    void doAbort();
    friend class ConnectionLife;
    Mutex syncLock;
    RWLock syncDisconnectLock;
    volatile bool stopped;
};

}

#endif
