#include <linux/module.h>
#include <linux/hrtimer.h>
#include <linux/of.h>
#include <linux/of_gpio.h>
#include <linux/pwm.h>
#include <linux/gpio.h>
#include <linux/platform_device.h>

struct gpio_pwm_device_data {
	struct hrtimer timer;
	struct gpio_desc *gpiod;
	bool inverted;
	bool pin_on;
	int on_time;
	int off_time;
	bool run;
};

struct gpio_pwm_priv {
	struct pwm_chip chip;
	struct gpio_pwm_device_data *pwm_device_data;
};

static void gpio_pwm_off(struct gpio_pwm_device_data *pwm_data)
{
	gpiod_set_value_cansleep(pwm_data->gpiod, pwm_data->inverted ? 1 : 0);
}

static void gpio_pwm_on(struct gpio_pwm_device_data *pwm_data)
{
	gpiod_set_value_cansleep(pwm_data->gpiod, pwm_data->inverted ? 0 : 1);
}

static enum hrtimer_restart gpio_pwm_timer(struct hrtimer *timer)
{
	struct gpio_pwm_device_data *pwm_data = container_of(timer,
						struct gpio_pwm_device_data,
						timer);

	if (!pwm_data->run) {
		gpio_pwm_off(pwm_data);
		pwm_data->pin_on = false;
		return HRTIMER_NORESTART;
	}

	if (pwm_data->pin_on) {
		hrtimer_forward_now(&pwm_data->timer, ns_to_ktime(pwm_data->off_time));
		gpio_pwm_off(pwm_data);
		pwm_data->pin_on = false;
	} else {
		hrtimer_forward_now(&pwm_data->timer, ns_to_ktime(pwm_data->on_time));
		gpio_pwm_on(pwm_data);
		pwm_data->pin_on = true;
	}

	return HRTIMER_RESTART;
}

static int gpio_pwm_config(struct pwm_chip *chip, struct pwm_device *pwm,
			   int duty_ns, int period_ns)
{
	struct gpio_pwm_priv *priv = container_of(chip, struct gpio_pwm_priv,
						  chip);
	struct gpio_pwm_device_data *pwm_data = &priv->pwm_device_data[pwm->hwpwm];

	pwm_data->on_time = duty_ns;
	pwm_data->off_time = period_ns - duty_ns;

	dev_info(chip->dev, "on_time: %d; off_time: %d; period_ns: %d\n",
		pwm_data->on_time, pwm_data->off_time, period_ns);

	return 0;
}

static int gpio_pwm_set_polarity(struct pwm_chip *chip, struct pwm_device *pwm,
				 enum pwm_polarity polarity)
{
	struct gpio_pwm_priv *priv = container_of(chip, struct gpio_pwm_priv,
						  chip);
	struct gpio_pwm_device_data *pwm_data = &priv->pwm_device_data[pwm->hwpwm];

	pwm_data->inverted = (polarity == PWM_POLARITY_INVERSED) ? true : false;

	return 0;
}

static int gpio_pwm_enable(struct pwm_chip *chip, struct pwm_device *pwm)
{
	struct gpio_pwm_priv *priv = container_of(chip, struct gpio_pwm_priv,
						  chip);
	struct gpio_pwm_device_data *pwm_data = &priv->pwm_device_data[pwm->hwpwm];

	if (pwm_data->run)
		return -EBUSY;

	pwm_data->run = true;
	if (pwm_data->off_time) {
		hrtimer_start(&pwm_data->timer, ktime_set(0, 0),
			      HRTIMER_MODE_REL);
	} else {
		if (pwm_data->on_time)
			gpio_pwm_on(pwm_data);
		else
			gpio_pwm_off(pwm_data);
	}

	return 0;
}

static void gpio_pwm_disable(struct pwm_chip *chip, struct pwm_device *pwm)
{
	struct gpio_pwm_priv *priv = container_of(chip, struct gpio_pwm_priv,
						  chip);
	struct gpio_pwm_device_data *pwm_data = &priv->pwm_device_data[pwm->hwpwm];

	pwm_data->run = false;
	if (!pwm_data->off_time)
		gpio_pwm_off(pwm_data);
}

static const struct pwm_ops gpio_pwm_ops = {
	.config = gpio_pwm_config,
	.set_polarity = gpio_pwm_set_polarity,
	.enable = gpio_pwm_enable,
	.disable = gpio_pwm_disable,
	.owner = THIS_MODULE,
};

static int gpio_pwm_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct gpio_pwm_priv *priv;
	int ret;
	unsigned int i;
	int npwm;

	if (!dev->of_node) {
		dev_err(dev, "missing DT info?\n");
		return -EINVAL;
	}

	npwm = of_gpio_named_count(dev->of_node, "pwm-gpios");
	if (npwm < 0) {
		dev_err(dev, "npwm: %d\n", npwm);
		return npwm;
	}

	if (npwm < 1) {
		dev_warn(dev, "gpios is empty\n");
		return -ENODEV;
	}

	priv = devm_kzalloc(dev, sizeof(*priv), GFP_KERNEL);
	if (!priv)
		return -ENOMEM;

	priv->pwm_device_data = devm_kzalloc(dev, npwm * sizeof(*priv->pwm_device_data), GFP_KERNEL);
	if (!priv->pwm_device_data)
		return -ENOMEM;

	priv->chip.dev = dev;
	priv->chip.ops = &gpio_pwm_ops;
	priv->chip.base = -1;
	priv->chip.npwm = npwm;
	priv->chip.of_xlate = of_pwm_xlate_with_flags;
	priv->chip.of_pwm_n_cells = 3;
	priv->chip.can_sleep = true;

	for (i = 0; i < npwm; ++i) {
		struct gpio_desc *gpiod;
		struct gpio_pwm_device_data *pwm_data = &priv->pwm_device_data[i];

		gpiod = devm_gpiod_get_index(&pdev->dev, "pwm", i, GPIOD_OUT_LOW);
		if (IS_ERR(gpiod)) {
			ret = PTR_ERR(gpiod);
			if (ret != -EPROBE_DEFER)
				dev_err(dev, "failed to get gpio: %d\n", ret);
			else
				dev_err(dev, "probe deferral\n");
			goto error;
		}

		hrtimer_init(&pwm_data->timer, CLOCK_MONOTONIC, HRTIMER_MODE_REL);
		pwm_data->timer.function = &gpio_pwm_timer;
		pwm_data->gpiod = gpiod;
		pwm_data->pin_on = false;
		pwm_data->run = false;

		if (!hrtimer_is_hres_active(&pwm_data->timer)) {
			dev_err(dev, "hrtimer is not high resolution :(\n");
			ret = -EINVAL;
			goto error;
		}
	}

	ret = pwmchip_add(&priv->chip);
	if (ret < 0) {
		dev_err(dev, "failed to add PWM chip: %d\n", ret);
		return ret;
	}

	platform_set_drvdata(pdev, priv);

	dev_info(dev, "%d gpios pwms loaded\n", npwm);

	return 0;

error:
	return ret;
}

static int gpio_pwm_remove(struct platform_device *pdev)
{
	struct gpio_pwm_priv *priv;
	unsigned int i;
	priv = platform_get_drvdata(pdev);
	if (!priv)
		return 0;

	for (i = 0; i < priv->chip.npwm; i++) {
		struct gpio_pwm_device_data *pwm_data;

		pwm_data = pwm_get_chip_data(&priv->chip.pwms[i]);
		hrtimer_cancel(&pwm_data->timer);
		gpiod_set_value_cansleep(pwm_data->gpiod, 0);
	}

	return pwmchip_remove(&priv->chip);
}

static const struct of_device_id gpio_pwm_of_match[] = {
	{ .compatible = "pwm-gpio", },
	{ /* sentinel */ },
};

static struct platform_driver gpio_pwm_driver = {
	.probe = gpio_pwm_probe,
	.remove = gpio_pwm_remove,
	.driver = {
		.name = "pwm-gpio",
		.of_match_table = gpio_pwm_of_match,
	},
};

module_platform_driver(gpio_pwm_driver);

MODULE_AUTHOR("Nicolae Rosia <Nicolae_Rosia@Mentor.com>");
MODULE_DESCRIPTION("Generic GPIO bit-banged PWM driver");
MODULE_LICENSE("GPL");
