#include <linux/err.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/pwm.h>
#include <linux/platform_device.h>
#include <linux/regulator/driver.h>
#include <linux/regulator/machine.h>
#include <linux/regulator/of_regulator.h>

struct pr_drvdata {
	struct mutex lock;
	struct pwm_device *pwm;
	bool is_enabled;
};

static ssize_t pr_show_name(struct device *dev,
			    struct device_attribute *attr, char *buf)
{
	return sprintf(buf, "%s\n", dev->of_node->name);
}

static ssize_t pr_set_state(struct device *dev, struct device_attribute *attr,
			    const char *buf, size_t count)
{
	struct pr_drvdata *drvdata = dev_get_drvdata(dev);
	bool enabled;
	int ret;

	/* serialize multiple writers */
	mutex_lock(&drvdata->lock);

	if (sysfs_streq(buf, "1")) {
		enabled = true;
	} else if (sysfs_streq(buf, "0")) {
		enabled = false;
	} else {
		dev_err(dev, "Expected to get 0 or 1\n");
		ret = -EINVAL;
		goto error;
	}

	if (enabled) {
		ret = pwm_enable(drvdata->pwm);
		if (ret) {
			dev_err(dev, "failed to enable pwm: %d\n", ret);
			goto error;
		}
	} else {
		pwm_disable(drvdata->pwm);
	}

	mutex_unlock(&drvdata->lock);
	return count;
error:
	mutex_unlock(&drvdata->lock);
	return ret;
}

static DEVICE_ATTR(name, 0444, pr_show_name, NULL);
static DEVICE_ATTR(state, 0644, NULL, pr_set_state);

static struct attribute *attributes[] = {
	&dev_attr_name.attr,
	&dev_attr_state.attr,
	NULL,
};

static const struct attribute_group attr_group = {
	.attrs	= attributes,
};

static int pr_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct pr_drvdata*drvdata;
	u32 duty_cycle;
	u32 period;
	int ret;

	if (!dev->of_node) {
		dev_err(dev, "missing DT info?\n");
		return -EINVAL;
	}

	drvdata = devm_kzalloc(dev, sizeof(*drvdata), GFP_KERNEL);
	if (!drvdata)
		return -ENOMEM;

	mutex_init(&drvdata->lock);

	ret = of_property_read_u32(dev->of_node, "mgc,duty-cycle", &duty_cycle);
	if (ret) {
		dev_warn(dev, "mgc,duty-cycle is required\n");
		return -EINVAL;
	}

	ret = of_property_read_u32(dev->of_node, "mgc,period", &period);
	if (ret) {
		dev_warn(dev, "mgc,duty-cycle is required\n");
		return -EINVAL;
	}

	drvdata->pwm = devm_of_pwm_get(dev, dev->of_node, NULL);
	if (IS_ERR(drvdata->pwm)) {
		ret = PTR_ERR(drvdata->pwm);
		dev_err(dev, "failed to get pwm: %d\n", ret);
		goto error;
	}
	ret = pwm_config(drvdata->pwm, duty_cycle, period);
	if (ret) {
		dev_err(dev, "failed to configure pwm: %d\n", ret);
		goto error;
	}

	platform_set_drvdata(pdev, drvdata);

	ret = sysfs_create_group(&pdev->dev.kobj, &attr_group);
	if (ret != 0)
		return ret;

	dev_info(dev, "pwm relay registered\n");

	return 0;

error:
	return ret;
}

static int pr_remove(struct platform_device *pdev)
{
	struct pr_drvdata *drvdata = platform_get_drvdata(pdev);

	if (drvdata) {
		sysfs_remove_group(&pdev->dev.kobj, &attr_group);
		pwm_disable(drvdata->pwm);
	}

	return 0;
}

static const struct of_device_id of_match[] = {
	{ .compatible = "mgc,pwm-relay", },
	{ /* sentinel */ },
};

static struct platform_driver pr_driver = {
	.probe = pr_probe,
	.remove = pr_remove,
	.driver = {
		.name = "pwm-relay",
		.of_match_table = of_match,
	},
};

module_platform_driver(pr_driver);

MODULE_AUTHOR("Nicolae Rosia <Nicolae_Rosia@Mentor.com>");
MODULE_DESCRIPTION("PWM Relay Driver");
MODULE_LICENSE("GPLv2");
