#include <linux/err.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/gpio.h>
#include <linux/hrtimer.h>
#include <linux/of.h>
#include <linux/of_gpio.h>
#include <linux/platform_device.h>
#include <linux/regulator/driver.h>
#include <linux/regulator/machine.h>
#include <linux/regulator/of_regulator.h>
#include <linux/atomic.h>

#define GPIO_SET (0)
#define GPIO_RST (1)
#define GPIO_PINS_COUNT (2)

#define FSM_STATE_IDLE (0)
#define FSM_STATE_BUSY_SET (1)
#define FSM_STATE_BUSY_RST (2)

/*
 * @gpios: SET and RST GPIO
 */
struct lr_drvdata {
	struct mutex lock;
	struct gpio_desc *gpios[GPIO_PINS_COUNT];
	unsigned int on_time;
	struct hrtimer timer;
	atomic_t state;
};

static ssize_t lr_show_name(struct device *dev,
			     struct device_attribute *attr, char *buf)
{
	return sprintf(buf, "%s\n", dev->of_node->name);
}

static ssize_t lr_set_state(struct device *dev, struct device_attribute *attr,
			     const char *buf, size_t count)
{
	struct lr_drvdata *drvdata = dev_get_drvdata(dev);
	bool enabled;
	int ret;

	/* serialize multiple writers */
	mutex_lock(&drvdata->lock);

	if (atomic_read(&drvdata->state) != FSM_STATE_IDLE) {
		ret = -EBUSY;
		goto error;
	} else {
		if (sysfs_streq(buf, "1")) {
			enabled = true;
		} else if (sysfs_streq(buf, "0")) {
			enabled = false;
		} else {
			dev_err(dev, "Expected to get 0 or 1\n");
			ret = -EINVAL;
			goto error;
		}
	}

	if (enabled) {
		atomic_set(&drvdata->state, FSM_STATE_BUSY_SET);
		gpiod_set_value_cansleep(drvdata->gpios[GPIO_SET], 1);
	} else {
		atomic_set(&drvdata->state, FSM_STATE_BUSY_RST);
		gpiod_set_value_cansleep(drvdata->gpios[GPIO_RST], 1);
	}

	hrtimer_start(&drvdata->timer, ns_to_ktime(drvdata->on_time),
		      HRTIMER_MODE_REL);

	mutex_unlock(&drvdata->lock);
	return count;

error:
	mutex_unlock(&drvdata->lock);
	return ret;
}

static DEVICE_ATTR(name, 0444, lr_show_name, NULL);
static DEVICE_ATTR(state, 0644, NULL, lr_set_state);

static struct attribute *attributes[] = {
	&dev_attr_name.attr,
	&dev_attr_state.attr,
	NULL,
};

static const struct attribute_group attr_group = {
	.attrs	= attributes,
};

static enum hrtimer_restart gpio_pulse_timer(struct hrtimer *timer)
{
	struct lr_drvdata *drvdata = container_of(timer, struct lr_drvdata,
						  timer);

	int state = atomic_read(&drvdata->state);

	if (state == FSM_STATE_BUSY_SET) {
		gpiod_set_value_cansleep(drvdata->gpios[GPIO_SET], 0);
	} else if (state == FSM_STATE_BUSY_RST) {
		gpiod_set_value_cansleep(drvdata->gpios[GPIO_RST], 0);
	}

	atomic_set(&drvdata->state, FSM_STATE_IDLE);

	return HRTIMER_NORESTART;
}

static int lr_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct lr_drvdata *drvdata;
	int ret;
	int ngpios;
	unsigned int i;

	if (!dev->of_node) {
		dev_err(dev, "missing DT info?\n");
		return -EINVAL;
	}

	ngpios = of_gpio_named_count(dev->of_node, "pulse-gpios");
	if (ngpios < 0) {
		dev_err(dev, "ngpios: %d\n", ngpios);
		return ngpios;
	}

	if (ngpios < 1) {
		dev_warn(dev, "gpios is empty\n");
		return -ENODEV;
	}

	if (ngpios != GPIO_PINS_COUNT) {
		dev_warn(dev, "gpios should contain (%d) entries\n", GPIO_PINS_COUNT);
		return -ENODEV;
	}

	drvdata = devm_kzalloc(dev, sizeof(*drvdata), GFP_KERNEL);
	if (!drvdata)
		return -ENOMEM;

	ret = of_property_read_u32(dev->of_node, "mgc,on-time", &drvdata->on_time);
	if (ret) {
		dev_warn(dev, "mgc,on-time is required\n");
		return -EINVAL;
	}

	mutex_init(&drvdata->lock);

	for (i = 0; i < GPIO_PINS_COUNT; ++i) {
		struct gpio_desc *gpiod;

		gpiod = devm_gpiod_get_index(&pdev->dev, "pulse", i, GPIOD_OUT_LOW);
		if (IS_ERR(gpiod)) {
			ret = PTR_ERR(gpiod);
			if (ret != -EPROBE_DEFER)
				dev_err(dev, "failed to get gpio: %d\n", ret);
			else
				dev_err(dev, "probe deferral\n");
			goto error;
		}

		drvdata->gpios[i] = gpiod;
	}

	hrtimer_init(&drvdata->timer, CLOCK_MONOTONIC, HRTIMER_MODE_REL);
	drvdata->timer.function = &gpio_pulse_timer;
	atomic_set(&drvdata->state, FSM_STATE_IDLE);

	if (!hrtimer_is_hres_active(&drvdata->timer)) {
		dev_err(dev, "hrtimer is not high resolution :(\n");
		ret = -EINVAL;
		goto error;
	}

	platform_set_drvdata(pdev, drvdata);

	ret = sysfs_create_group(&pdev->dev.kobj, &attr_group);
	if (ret != 0)
		return ret;

	dev_info(dev, "latch relay registered\n");

	return 0;

error:
	return ret;
}

static int lr_remove(struct platform_device *pdev)
{
	struct lr_drvdata *drvdata = platform_get_drvdata(pdev);

	if (drvdata) {
		sysfs_remove_group(&pdev->dev.kobj, &attr_group);

		hrtimer_cancel(&drvdata->timer);

		gpiod_set_value_cansleep(drvdata->gpios[GPIO_SET], 0);
		gpiod_set_value_cansleep(drvdata->gpios[GPIO_RST], 0);
	}

	return 0;
}

static const struct of_device_id of_match[] = {
	{ .compatible = "mgc,latch-relay", },
	{ /* sentinel */ },
};

static struct platform_driver lr_driver = {
	.probe = lr_probe,
	.remove = lr_remove,
	.driver = {
		.name = "latch-relay",
		.of_match_table = of_match,
	},
};

module_platform_driver(lr_driver);

MODULE_AUTHOR("Nicolae Rosia <Nicolae_Rosia@Mentor.com>");
MODULE_DESCRIPTION("Latch Relay Driver");
MODULE_LICENSE("GPLv2");
