/*
 * Copyright 2011 Google Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.google.devtools.j2objc;

import com.google.devtools.j2objc.Options.OutputStyleOption;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

/**
 * Tests for {@link com.google.devtools.j2objc.J2ObjC#run(List)}.
 */
public class J2ObjCTest extends GenerationTest {
  String jarPath;
  String exampleJavaPath;
  String packageInfoPath;

  private static final String EXAMPLE_JAVA_SOURCE =
      "package com.google.devtools.j2objc.annotations;\n\n"
      + "import com.google.j2objc.annotations.ObjectiveCName;\n\n"
      + "@ObjectiveCName(\"foo\")\n"
      + "public class Example {\n"
      + "}\n";

  @Override
  public void setUp() throws IOException {
    super.setUp();

    List<String> classpathEntries = Options.getClassPathEntries();
    for (String entry : getComGoogleDevtoolsJ2objcPath()) {
      classpathEntries.add(entry);
    }

    jarPath = getResourceAsFile("util/example.jar");
    exampleJavaPath = "com/google/devtools/j2objc/util/Example.java";
    packageInfoPath = "com/google/devtools/j2objc/util/package-info.java";
    addSourceFile(
        "package com.google.devtools.j2objc.util; public class Example {}", exampleJavaPath);
    addSourceFile(
        "@ObjectiveCName(\"CBT\")\n"
        + "package com.google.devtools.j2objc.util;\n"
        + "import com.google.j2objc.annotations.ObjectiveCName;", packageInfoPath);
  }

  @Override
  public void tearDown() throws Exception {
    Options.getClassPathEntries().clear();
    Options.setBatchTranslateMaximum(0);
    super.tearDown();
  }

  // No assertions for packageInfoH--nothing interesting in it that other tests don't assert.
  private void makeAssertions(String exampleH, String exampleM, String packageInfoM)
      throws Exception {
    // These assertions should hold for any generated files, including (in the future)
    // combined jar generation.
    assertTranslation(exampleH, "Generated by the J2ObjC translator.");
    assertTranslation(exampleH, "interface CBTExample : NSObject");
    assertTranslation(exampleH, "- (instancetype)init;");
    assertTranslation(exampleH, "J2OBJC_EMPTY_STATIC_INIT(CBTExample)");
    assertTranslation(exampleH, "J2OBJC_TYPE_LITERAL_HEADER(CBTExample)");
    assertTranslation(exampleM, "@implementation CBTExample");
    assertTranslation(exampleM, "- (instancetype)init {");
    assertTranslation(exampleM, "+ (const J2ObjcClassInfo *)__metadata {");
    assertTranslation(exampleM, "J2OBJC_CLASS_TYPE_LITERAL_SOURCE(CBTExample)");
    assertTranslatedLines(packageInfoM,
        "+ (NSString *)__prefix {",
        "  return @\"CBT\";",
        "}");

    assertErrorCount(0);
  }

  private void makeAssertionsForJar() throws Exception {
    String exampleH = getTranslatedFile("com/google/test/Example.h");
    String exampleM = getTranslatedFile("com/google/test/Example.m");
    String packageInfoH = getTranslatedFile("com/google/test/package-info.h");
    String packageInfoM = getTranslatedFile("com/google/test/package-info.m");
    // Test the file header comments.
    assertTranslation(exampleH, "source: jar:file:");
    assertTranslation(exampleM, "source: jar:file:");
    assertTranslation(packageInfoH, "source: jar:file:");
    assertTranslation(packageInfoM, "source: jar:file:");
    assertTranslation(exampleH, jarPath);
    assertTranslation(exampleM, jarPath);
    assertTranslation(packageInfoH, jarPath);
    assertTranslation(packageInfoM, jarPath);
    assertTranslation(exampleH, "com/google/test/Example.java");
    assertTranslation(exampleM, "com/google/test/Example.java");
    assertTranslation(packageInfoH, "com/google/test/package-info.java");
    assertTranslation(packageInfoM, "com/google/test/package-info.java");
    // Test the includes
    assertTranslation(exampleH, "#pragma push_macro(\"INCLUDE_ALL_ComGoogleTestExample\")");
    assertTranslation(exampleM, "#include \"com/google/test/Example.h\"");
    assertTranslation(
        packageInfoH, "#pragma push_macro(\"INCLUDE_ALL_ComGoogleTestPackage_info\")");
    assertTranslation(packageInfoM, "@interface ComGoogleTestpackage_info : NSObject");
    assertTranslation(packageInfoM, "@implementation ComGoogleTestpackage_info");
    // All other assertions
    makeAssertions(exampleH, exampleM, packageInfoM);
  }

  public void testCompilingFromJar() throws Exception {
    J2ObjC.run(Collections.singletonList(jarPath));
    makeAssertionsForJar();
  }

  public void testBatchCompilingFromJar() throws Exception {
    Options.setBatchTranslateMaximum(2);
    J2ObjC.run(Collections.singletonList(jarPath));
    makeAssertionsForJar();
  }

  // Make assertions for java files with default output locations.
  private void makeAssertionsForJavaFiles() throws Exception {
    String exampleH = getTranslatedFile("com/google/devtools/j2objc/util/Example.h");
    String exampleM = getTranslatedFile("com/google/devtools/j2objc/util/Example.m");
    String packageInfoH = getTranslatedFile("com/google/devtools/j2objc/util/package-info.h");
    String packageInfoM = getTranslatedFile("com/google/devtools/j2objc/util/package-info.m");
    assertTranslation(exampleM, "#include \"com/google/devtools/j2objc/util/Example.h\"");
    makeAssertionsForJavaFiles(exampleH, exampleM, packageInfoH, packageInfoM);
  }

  private void makeAssertionsForJavaFiles(
      String exampleH, String exampleM, String packageInfoH, String packageInfoM)
      throws Exception {
    // Test the file header comments.
    assertTranslation(exampleH, exampleJavaPath);
    assertTranslation(exampleM, exampleJavaPath);
    assertTranslation(packageInfoH, packageInfoPath);
    assertTranslation(packageInfoM, packageInfoPath);
    // Test the includes
    assertTranslation(
        exampleH, "#pragma push_macro(\"INCLUDE_ALL_ComGoogleDevtoolsJ2objcUtilExample\")");
    assertTranslation(
        packageInfoM, "@interface ComGoogleDevtoolsJ2objcUtilpackage_info : NSObject");
    assertTranslation(packageInfoH,
        "#pragma push_macro(\"INCLUDE_ALL_ComGoogleDevtoolsJ2objcUtilPackage_info\")");
    assertTranslation(packageInfoM, "@implementation ComGoogleDevtoolsJ2objcUtilpackage_info");
    // All other assertions
    makeAssertions(exampleH, exampleM, packageInfoM);
  }

  public void testCompilingFromFiles() throws Exception {
    J2ObjC.run(Arrays.asList(exampleJavaPath, packageInfoPath));
    makeAssertionsForJavaFiles();
  }

  public void testBatchCompilingFromFiles() throws Exception {
    Options.setBatchTranslateMaximum(2);
    J2ObjC.run(Arrays.asList(exampleJavaPath, packageInfoPath));
    makeAssertionsForJavaFiles();
  }

  private void makeAssertionsForCombinedJar() throws Exception {
    String combinedH = getTranslatedFile(jarPath.replace(".jar", ".h"));
    String combinedM = getTranslatedFile(jarPath.replace(".jar", ".m"));
    makeAssertions(combinedH, combinedM, combinedM);
  }

  public void testCombinedJar() throws Exception {
    Options.setOutputStyle(OutputStyleOption.SOURCE_COMBINED);
    J2ObjC.run(Collections.singletonList(jarPath));
    makeAssertionsForCombinedJar();
  }

  public void testSourceDirsOption() throws Exception {
    Options.setOutputStyle(Options.OutputStyleOption.SOURCE);
    J2ObjC.run(Arrays.asList(exampleJavaPath, packageInfoPath));
    String exampleH = getTranslatedFile(exampleJavaPath.replace(".java", ".h"));
    String exampleM = getTranslatedFile(exampleJavaPath.replace(".java", ".m"));
    String packageInfoH = getTranslatedFile(packageInfoPath.replace(".java", ".h"));
    String packageInfoM = getTranslatedFile(packageInfoPath.replace(".java", ".m"));
    makeAssertionsForJavaFiles(exampleH, exampleM, packageInfoH, packageInfoM);
  }

  // Test a simple annotation processor on the classpath.
  public void testAnnotationProcessing() throws Exception {
    String processorPath = getResourceAsFile("annotations/Processor.jar");
    Options.getClassPathEntries().add(processorPath);

    String examplePath = addSourceFile(EXAMPLE_JAVA_SOURCE, "annotations/Example.java");
    J2ObjC.run(Collections.singletonList(examplePath));
    assertErrorCount(0);


    String translatedAnnotationHeader = getTranslatedFile("ProcessingResult.h");
    String translatedAnnotationImpl = getTranslatedFile("ProcessingResult.m");

    // Our dummy annotation processor is very simple--it always creates a class with no package,
    // ProcessingResult, with a minimal implementation.
    assertTranslation(translatedAnnotationHeader, "@interface ProcessingResult : NSObject");
    assertTranslation(translatedAnnotationHeader, "- (NSString *)getResult;");
    assertTranslation(translatedAnnotationImpl, "@implementation ProcessingResult");
    assertTranslation(translatedAnnotationImpl, "return @\"ObjectiveCName\"");
  }

  // Test a simple annotation processor on the processor path.
  public void testAnnotationProcessingWithProcessorPath() throws Exception {
    String processorPath = getResourceAsFile("annotations/Processor.jar");
    Options.getProcessorPathEntries().add(processorPath);

    String examplePath = addSourceFile(EXAMPLE_JAVA_SOURCE, "annotations/Example.java");
    J2ObjC.run(Collections.singletonList(examplePath));
    assertErrorCount(0);
  }

  // Test a specified annotation processor.
  public void testSpecifiedAnnotationProcessing() throws Exception {
    String processorPath = getResourceAsFile("annotations/Processor.jar");
    Options.getClassPathEntries().add(processorPath);
    Options.setProcessors("com.google.devtools.j2objc.annotations.J2ObjCTestProcessor");

    String examplePath = addSourceFile(EXAMPLE_JAVA_SOURCE, "annotations/Example.java");
    J2ObjC.run(Collections.singletonList(examplePath));
    assertErrorCount(0);
  }
}
